import ujson
from marshmallow import ValidationError, fields, post_dump, post_load, pre_load

from mail.ciao.ciao.api.schemas.fields import DateTimeTz
from mail.ciao.ciao.api.schemas.megamind.base import BaseMegamindSchema
from mail.ciao.ciao.api.schemas.scenario.create_event import CreateEventParamsSchema
from mail.ciao.ciao.api.schemas.scenario.delete_event import DeleteEventParamsSchema
from mail.ciao.ciao.api.schemas.scenario.event_list import EventListParamsSchema
from mail.ciao.ciao.api.schemas.scenario.find_event import FindEventParamsSchema
from mail.ciao.ciao.api.schemas.scenario.reschedule_event import RescheduleEventParamsSchema
from mail.ciao.ciao.core.scenarios.create_event import CreateEventScenario
from mail.ciao.ciao.core.scenarios.delete_event import DeleteEventScenario
from mail.ciao.ciao.core.scenarios.event_list import EventListScenario
from mail.ciao.ciao.core.scenarios.find_event import FindEventScenario
from mail.ciao.ciao.core.scenarios.reschedule_event import RescheduleEventScenario
from mail.ciao.ciao.utils.logging import LOGGER


class StateStackItemSchema(BaseMegamindSchema):
    _PARAMS_SCHEMAS = {
        scenario_cls.scenario_name: params_schema()  # type: ignore
        for scenario_cls, params_schema in (
            (CreateEventScenario, CreateEventParamsSchema),
            (DeleteEventScenario, DeleteEventParamsSchema),
            (EventListScenario, EventListParamsSchema),
            (FindEventScenario, FindEventParamsSchema),
            (RescheduleEventScenario, RescheduleEventParamsSchema),
        )
    }

    scenario_name = fields.String(required=True)
    params = fields.Field(required=True)
    arg_name = fields.String(allow_none=True)

    @post_dump(pass_original=True)
    def dump_params(self, data, obj):
        scenario_name = self.get_attribute('scenario_name', data, default=None)
        params_schema = self._PARAMS_SCHEMAS.get(scenario_name)
        params_data = self.get_attribute('params', data, {})
        data.pop('params', None)
        if params_schema is not None:
            data['params'], _ = params_schema.dump(params_data)
        return data

    @post_load
    def load_params(self, data):
        scenario_name = data['scenario_name']
        if scenario_name not in self._PARAMS_SCHEMAS:
            raise ValidationError(f'No params schema for scenario "{scenario_name}".')
        params_schema = self._PARAMS_SCHEMAS[scenario_name]
        params_data = data.pop('params', {})
        data['params'] = params_schema.load(params_data).data
        return data


class StateStackSchema(BaseMegamindSchema):
    stack_items = fields.Nested(StateStackItemSchema, many=True)


class StateSchema(BaseMegamindSchema):
    """
    This schema must never raise ValidationErrors since data could belong to a different service.
    """

    hard_expire = DateTimeTz(required=True)
    soft_expire = DateTimeTz(required=True)
    state_stack = fields.Nested(StateStackSchema)

    def load(self, *args, **kwargs):
        try:
            return super().load(*args, **kwargs)
        except ValidationError as exc:
            logger = LOGGER.get()
            with logger:
                logger.context_push(state_errors=str(exc))
                logger.info('State schema error.')
            return {}, None

    @pre_load
    def json_data_load(self, data):
        if not self.protobuf_context:
            return data
        # Undoing protobuf-specific packing in `json_data_dump`.
        try:
            return ujson.loads(data.get('json_data'))
        except (ValueError, TypeError):
            return {}

    @post_dump
    def json_data_dump(self, data):
        if not self.protobuf_context:
            return data
        # Protobuf requires `@type` for data to be used as "Any" message.
        return {
            '@type': 'type.googleapis.com/NMail.NCiao.TState',
            'json_data': ujson.dumps(data),
        }
