from dataclasses import dataclass
from datetime import datetime, timedelta
from typing import Union

from sendr_utils import utcnow

from mail.ciao.ciao.conf import settings
from mail.ciao.ciao.core.entities.missing import MissingType
from mail.ciao.ciao.core.entities.state_stack import StateStack


@dataclass
class State:
    # Set once when state is initialized.
    hard_expire: Union[MissingType, datetime] = MissingType.MISSING
    # Updated every time instance is created unless it's already expired.
    soft_expire: Union[MissingType, datetime] = MissingType.MISSING

    state_stack: Union[MissingType, StateStack] = MissingType.MISSING

    def __post_init__(self) -> None:
        self._set_hard_expire()
        self._set_soft_expire(force=not self.expired)

        if self.state_stack is MissingType.MISSING:
            self.state_stack = StateStack()

    def _set_hard_expire(self) -> None:
        if self.hard_expire is MissingType.MISSING:
            self.hard_expire = utcnow() + timedelta(seconds=settings.STATE_HARD_TTL)

    def _set_soft_expire(self, force: bool = False) -> None:
        if self.soft_expire is MissingType.MISSING or force:
            self.soft_expire = utcnow() + timedelta(seconds=settings.STATE_SOFT_TTL)

    @property
    def expired(self) -> bool:
        assert self.hard_expire is not MissingType.MISSING
        now = utcnow()
        return now >= self.hard_expire or (self.soft_expire is not MissingType.MISSING and now >= self.soft_expire)
