from dataclasses import dataclass, field
from typing import TYPE_CHECKING, Any, List, Optional, Tuple

if TYPE_CHECKING:
    from mail.ciao.ciao.core.scenarios.base import BaseScenario  # noqa


@dataclass
class StateStackItem:
    scenario_name: str
    params: dict = field(default_factory=dict)
    arg_name: Optional[str] = None
    scenario_instance: Optional['BaseScenario'] = None


class StateStack:
    _stack_items: List[StateStackItem]

    def __init__(self, stack_items: Optional[List[StateStackItem]] = None):
        self._stack_items = stack_items or []

    def __eq__(self, other: Any) -> bool:
        return isinstance(other, StateStack) and self.stack_items == other.stack_items

    def __repr__(self) -> str:
        stack_items = ', '.join([repr(stack_item) for stack_item in self._stack_items])
        return f'StateStack(stack_items=[{stack_items}])'

    @property
    def stack_items(self) -> Tuple[StateStackItem, ...]:
        return tuple(self._stack_items)

    @property
    def top(self) -> Optional[StateStackItem]:
        return next(reversed(self._stack_items), None)

    @property
    def empty(self) -> bool:
        return not bool(self._stack_items)

    def append(self, scenario: 'BaseScenario', arg_name: Optional[str] = None) -> None:
        self._stack_items.append(StateStackItem(
            scenario_name=scenario.scenario_name,
            params=scenario.get_params(),
            arg_name=arg_name,
        ))

    def pop(self) -> StateStackItem:
        top = self.top
        assert top is not None
        self._stack_items.pop()
        return top
