import asyncio
from typing import ClassVar, Dict, Type

from sendr_qlog import LoggerContext

from mail.ciao.ciao.conf import settings
from mail.ciao.ciao.core.context import CORE_CONTEXT, CoreContext
from mail.ciao.ciao.core.entities.enums import FrameName
from mail.ciao.ciao.core.entities.missing import MissingType
from mail.ciao.ciao.core.entities.scenario_response import IRRELEVANT_RESPONSE, ScenarioResponse
from mail.ciao.ciao.core.entities.scenario_result import ScenarioResult
from mail.ciao.ciao.core.entities.state import State
from mail.ciao.ciao.core.entities.state_stack import StateStack
from mail.ciao.ciao.core.exceptions import CoreIrrelevantScenarioError
from mail.ciao.ciao.core.scenarios import SCENARIO_BY_NAME, BaseScenario, CreateEventScenario, EventListScenario


class ScenarioRunner:
    """
    Runs scenarios on state stack.
    """
    context: ClassVar[CoreContext] = CORE_CONTEXT

    _STARTER_SCENARIO_BY_FRAME: Dict[FrameName, Type[BaseScenario]] = {
        FrameName.EVENT_LIST: EventListScenario,
        FrameName.CREATE_EVENT: CreateEventScenario,
    }
    _MAX_ITERATIONS = settings.SCENARIO_RUNNER_MAX_ITERATIONS

    def __init__(self, frame_name: FrameName, slots: dict, commit: bool):
        self._frame_name: FrameName = frame_name
        self._slots: dict = slots
        self._commit: bool = commit

    @property
    def logger(self) -> LoggerContext:
        return self.context.logger

    @property
    def state(self) -> State:
        assert self.context.state is not None
        return self.context.state

    @property
    def state_stack(self) -> StateStack:
        assert self.state.state_stack is not MissingType.MISSING
        return self.state.state_stack

    async def _run_top_scenario(self, pass_params: bool = False) -> ScenarioResult:
        stack_top = self.state_stack.top
        assert stack_top is not None
        self.logger.context_push(sceanrio_name=stack_top.scenario_name)
        try:
            scenario_cls = SCENARIO_BY_NAME[stack_top.scenario_name]
            scenario_instance = stack_top.scenario_instance = scenario_cls(**dict(
                **stack_top.params,
                frame_name=self._frame_name if pass_params else None,
                slots=self._slots if pass_params else None,
                commit=self._commit and pass_params,
            ))
            self.logger.info(f'Running scenarion "{stack_top.scenario_name}".')
            result = await scenario_instance.run()
            stack_top.params = scenario_instance.get_params()
            return result
        except asyncio.CancelledError:
            self.logger.info('Cancelled error.')
            raise
        except CoreIrrelevantScenarioError as e:
            self.logger.info(f'Irrelevant scenario error: "{e}".')
            return ScenarioResult(response=ScenarioResponse(irrelevant=True, analytics=e.analytics))
        except Exception as e:
            self.logger.exception('Unhandled scenario error.')
            return ScenarioResult(
                response=ScenarioResponse(error=str(type(e))),
            )

    async def _handle_irrelevant_response(self, result: ScenarioResult) -> bool:
        # Scenario returned irrelevant response. We need to try running the previous scenario which may be able to
        # handle this request.
        if result.response is None or not result.response.irrelevant:
            return False
        self.state_stack.pop()
        return True

    async def _handle_call(self, result: ScenarioResult) -> bool:
        # Scenario called another scenario. We need to put it on stack and call it during the next iteration.
        if result.call is None:
            return False
        self.state_stack.append(
            scenario=result.call[0],
            arg_name=result.call[1],
        )
        return True

    async def _handle_value(self, result: ScenarioResult) -> bool:
        # Scenario finished running and returned value. Previous scenario must be called with the result of the current
        # scenario. In case there are no other scenarios on stack, response must be returned.
        if result.value is MissingType.MISSING:
            return False
        arg_name = self.state_stack.pop().arg_name
        top = self.state_stack.top
        if top is not None:
            if arg_name is not None:
                top.params[arg_name] = result
            return True
        return False

    async def _handle_relevant_response(self, result: ScenarioResult) -> bool:
        # Scenario returned relevant response.
        return result.response is not None and not result.response.irrelevant

    async def _run(self) -> ScenarioResponse:
        """Executes top scenario until stack is empty. Result is handled by `_handle_*` methods. These methods return
        `True` if the result has been handled and no following handle methods must be called. Handle methods might
        modify stack and must always be called in correct order.

        Params `frame_name` and `slots` must only be passed to the first scenario that is able to handle them. Meaning
        once the scenario that does not return irrelevant response is found, `frame_name` and `slots` are not passed to
        scenarios anymore.
        """
        entry_found: bool = False
        iteration = 0
        while not self.state_stack.empty and iteration < self._MAX_ITERATIONS:
            iteration += 1

            result = await self._run_top_scenario(pass_params=not entry_found)
            if await self._handle_irrelevant_response(result):
                if entry_found:
                    self.logger.error('Scenario returned irrelevant response after the entry was found.')
                    entry_found = False
                else:
                    self.logger.info('Scenario returned irrelevant response.')
            else:
                entry_found = True
                if await self._handle_call(result):
                    self.logger.info('Scenario called antoher.')
                elif await self._handle_value(result):
                    self.logger.info('Scenario finished.')
                elif await self._handle_relevant_response(result):
                    self.logger.info('Scenario returned relevant response.')
                    assert result.response is not None
                    return result.response
                else:
                    self.logger.error('Scenario returned invalid result.')
                    break

        if iteration >= self._MAX_ITERATIONS:
            self.logger.error('ScenarioRunner max iterations limit exceeded.')
        return IRRELEVANT_RESPONSE

    async def run(self) -> ScenarioResponse:
        response = await self._run()
        if not response.irrelevant:
            return response

        assert self.state_stack.empty
        starter_scenario_cls = self._STARTER_SCENARIO_BY_FRAME.get(self._frame_name)
        if starter_scenario_cls is not None:
            self.state_stack.append(scenario=starter_scenario_cls())
            with self.logger:
                return await self._run()

        self.logger.info('Starter scenario was not found.')
        return response
