from abc import ABCMeta, abstractmethod
from typing import ClassVar, Generic, Optional, Type, TypeVar, Union, overload

from mail.ciao.ciao.core.actions.base import BaseAction
from mail.ciao.ciao.core.entities.enums import FrameName
from mail.ciao.ciao.core.entities.missing import MissingType
from mail.ciao.ciao.core.entities.scenario_result import ScenarioResult
from mail.ciao.ciao.core.entities.user import User
from mail.ciao.ciao.core.exceptions import CoreIrrelevantScenarioError
from mail.ciao.ciao.utils.logging import SensitiveDataHolder
from mail.ciao.ciao.utils.stats import scenario_request_count, scenario_request_time

T = TypeVar('T')
T2 = TypeVar('T2')
SLOT_TYPE = TypeVar('SLOT_TYPE')


class BaseScenario(BaseAction, Generic[T], metaclass=ABCMeta):
    scenario_name: ClassVar[str]

    def __init__(self,
                 frame_name: Optional[FrameName] = None,
                 slots: Optional[dict] = None,
                 commit: bool = False,
                 ):
        super().__init__()
        self._frame_name: Optional[FrameName] = frame_name
        self._slots: Optional[dict] = slots
        self._commit: bool = commit

    @property
    def user(self) -> User:
        assert self.context.user is not None
        return self.context.user

    @property
    def stack(self):
        return self.context.state.state_stack

    @property
    def slots(self) -> dict:
        if self._slots is None:
            return {}
        return self._slots

    @abstractmethod
    def _get_params(self) -> dict:
        raise NotImplementedError

    def get_params(self) -> dict:
        params = self._get_params()
        assert 'frame_name' not in params and 'slots' not in params and 'commit' not in params
        return params

    @overload
    def get_slot(self, slot_name: str, slot_type: Type[SLOT_TYPE]) -> Union[SLOT_TYPE, MissingType]:
        ...

    @overload  # noqa: RedefinedWhileUnused
    def get_slot(self,
                 slot_name: str,
                 slot_type: Type[SLOT_TYPE],
                 default: T2,
                 ) -> Union[SLOT_TYPE, T2]:
        ...

    def get_slot(self, slot_name, slot_type, default=MissingType.MISSING):  # noqa: RedefinedWhileUnused
        value = self.slots.get(slot_name, MissingType.MISSING)
        if value is MissingType.MISSING or not isinstance(value, slot_type):
            return default
        return value

    def require_slot(self, slot_name: str, slot_type: Type[SLOT_TYPE]) -> SLOT_TYPE:
        value = self.get_slot(slot_name, slot_type)
        if value is MissingType.MISSING:
            raise CoreIrrelevantScenarioError(
                f'Scenario did not receive required slot "{slot_name}" of type "{slot_type}")'
            )
        return value

    @abstractmethod
    async def handle(self) -> ScenarioResult[T]:
        raise NotImplementedError

    async def run(self) -> ScenarioResult[T]:
        commit_label = 'commit' if self._commit else 'no_commit'

        with self.logger, scenario_request_time.labels(self.scenario_name, commit_label).time:
            scenario_request_count.labels(self.scenario_name, commit_label).inc()

            stack_top = self.stack.top
            assert stack_top is not None and stack_top.scenario_instance is self, \
                'Scenario can only be run while being on top of the state stack.'
            self.logger.context_push(
                scenario_frame_name=self._frame_name,
                scenario_slots=SensitiveDataHolder(self._slots),
                scenario_commit=self._commit,
            )

            self.logger.info('Scenario execution start.')
            result = await super().run()
            self.logger.info('Scenario execution end.')

            return result
