from datetime import datetime, timedelta
from enum import Enum, unique
from itertools import product
from typing import Any, ClassVar, Optional, Set, Tuple

from dateutil.relativedelta import relativedelta

from sendr_utils import enum_value

from mail.ciao.ciao.conf import settings
from mail.ciao.ciao.core.entities.analytcs import Action, Analytics, Intent
from mail.ciao.ciao.core.entities.button import get_calendar_day_button
from mail.ciao.ciao.core.entities.enums import CreateEventAllDay, FrameName, SysSlotType, YesNo
from mail.ciao.ciao.core.entities.missing import MissingType
from mail.ciao.ciao.core.entities.scenario_response import RequestedSlotType, ScenarioResponse
from mail.ciao.ciao.core.entities.scenario_result import ScenarioResult
from mail.ciao.ciao.core.exceptions import CoreIrrelevantScenarioError
from mail.ciao.ciao.core.scenarios.base import BaseScenario
from mail.ciao.ciao.utils.datetime import Period, UserDate, UserTime, timezone_today
from mail.ciao.ciao.utils.format import format_date, format_datetime, format_time
from mail.ciao.ciao.utils.gettext import gettext
from mail.ciao.ciao.utils.stats import scenario_event_create_count, scenario_event_create_time


class EventEndError(Exception):  # Internal exception, not propagated
    pass


@unique
class CreateEventScenarioState(Enum):
    INITIAL = 'initial'
    FILLING_SLOTS = 'filling_slots'
    AWAITING_CONFIRMATION = 'awaiting_confirmation'


class CreateEventScenario(BaseScenario):
    """
    https://wiki.yandex-team.ru/mail/swat/ciao/#m-dobavleniesobytija
    """
    scenario_name = 'create_event'

    _SUBSEQUENT_FRAME: ClassVar[FrameName] = FrameName.CREATE_EVENT_SUBSEQUENT

    _EVENT_NAME_SLOT: ClassVar[str] = 'create_event_event_name'
    _DATE_START_SLOT: ClassVar[str] = 'create_event_date_start'
    _DATE_END_SLOT: ClassVar[str] = 'create_event_date_end'
    _TIME_START_SLOT: ClassVar[str] = 'create_event_time_start'
    _TIME_END_SLOT: ClassVar[str] = 'create_event_time_end'
    _DURATION_SLOT: ClassVar[str] = 'create_event_duration'
    _CONFIRMATION_SLOT: ClassVar[str] = 'create_event_confirmation'
    _ALL_DAY_SLOT: ClassVar[str] = 'create_event_all_day'

    def __init__(self,
                 *args: Any,
                 state: CreateEventScenarioState = CreateEventScenarioState.INITIAL,
                 date_start: Optional[UserDate] = None,
                 time_start: Optional[UserTime] = None,
                 date_end: Optional[UserDate] = None,
                 time_end: Optional[UserTime] = None,
                 duration: Optional[relativedelta] = None,
                 event_name: Optional[str] = None,
                 all_day: bool = False,
                 **kwargs: Any,
                 ):
        super().__init__(*args, **kwargs)
        self._state = state

        # slots
        self._date_start = date_start
        self._time_start = time_start
        self._date_end = date_end
        self._time_end = time_end
        self._duration = duration
        self._event_name = event_name
        self._all_day = all_day

    def _get_params(self) -> dict:
        return {
            'state': self._state,
            'date_start': self._date_start,
            'time_start': self._time_start,
            'date_end': self._date_end,
            'time_end': self._time_end,
            'duration': self._duration,
            'event_name': self._event_name,
            'all_day': self._all_day,
        }

    @classmethod
    def _build_request(cls,
                       text: str,
                       requested_slot: RequestedSlotType,
                       contains_sensitive_data: bool = False,
                       ) -> ScenarioResult:
        """
        Builds ScenarioResult with slot request.
        """
        return ScenarioResult(response=ScenarioResponse(
            text=text,
            frame_name=cls._SUBSEQUENT_FRAME,
            requested_slot=requested_slot,
            contains_sensitive_data=contains_sensitive_data,
        ))

    def _update_from_slots(self) -> None:
        """
        Goes through all known slots and updates fields with them unless they're already set.
        """
        for slot_name, slot_type, attr_name in (
            (self._DATE_START_SLOT, UserDate, '_date_start'),
            (self._TIME_START_SLOT, UserTime, '_time_start'),
            (self._DATE_END_SLOT, UserDate, '_date_end'),
            (self._TIME_END_SLOT, UserTime, '_time_end'),
            (self._DURATION_SLOT, relativedelta, '_duration'),
            (self._EVENT_NAME_SLOT, str, '_event_name'),
        ):
            slot_value = self.get_slot(slot_name, slot_type)
            if getattr(self, attr_name, None) is None and slot_value is not MissingType.MISSING:
                setattr(self, attr_name, slot_value)

        if self._event_name is not None:
            self._event_name = self._event_name.capitalize()

        self._all_day = self._all_day or (
            self.get_slot(self._ALL_DAY_SLOT, CreateEventAllDay) is CreateEventAllDay.ALL_DAY)

    def _request_missing_slot(self, analytics: Analytics) -> Optional[ScenarioResult]:
        """
        Finds first missing required slot and returns request for it. If all required slots are present, returns None.
        """
        # In case time_start is relative, we can skip date_start.
        if self._date_start is None and (self._time_start is None or not self._time_start.relative):
            analytics.actions.append(Action.CREATE_EVENT_START_DATE)
            return self._build_request(
                text=gettext('What day must event start?'),
                requested_slot=(self._DATE_START_SLOT, SysSlotType.DATE),
            )
        # If all_day is passed, time_start is redundant.
        if self._time_start is None and not self._all_day:
            analytics.actions.append(Action.CREATE_EVENT_START_TIME)
            return self._build_request(
                text=gettext('What time must event start?'),
                requested_slot=(self._TIME_START_SLOT, SysSlotType.TIME),
            )
        # If all_day is passed, duration is redundant.
        if self._time_end is None and self._duration is None and not self._all_day:
            analytics.actions.append(Action.CREATE_EVENT_DURATION)
            return self._build_request(
                text=gettext('How long must event be?'),
                requested_slot=(self._DURATION_SLOT, SysSlotType.DATETIME_RANGE),
            )
        if self._event_name is None:
            analytics.actions.append(Action.CREATE_EVENT_NAME)
            return self._build_request(
                text=gettext('What is the name of the event?'),
                requested_slot=(self._EVENT_NAME_SLOT, SysSlotType.STRING),
            )
        return None

    def _get_event_start_end(self) -> Tuple[datetime, datetime]:
        """
        Tries to guess intended event start and end times. If all_day is set, uses only start date. Otherwise requires
        event start date, start time and one of: end time, duration. If duration is present uses it, otherwise uses end
        date and time. If end date is missing, uses either today or tomorrow so that end datetime is after start
        datetime.
        """
        today = timezone_today(self.user.timezone)
        now = datetime.now(self.user.timezone)

        if self._all_day:
            assert self._date_start is not None
            datetime_start = self.user.timezone.localize(datetime.combine(
                self._date_start.get_date(today),
                datetime.min.time(),
            ))
            return datetime_start, datetime_start + timedelta(days=1)

        assert self._time_start is not None and (self._date_start is not None or self._time_start.relative)

        possible_start: Set[datetime] = set()
        possible_end: Set[datetime] = set()

        if self._time_start.relative:
            possible_start = set((self._time_start.get_absolute(now),))
        else:
            assert self._date_start is not None
            if self._time_end is None or not self._time_end.fixed:
                # In case no fixed time mentioned, fitting start time to work day
                possible_time_start = set((self._time_start.get_fit_time(settings.WORK_DAY_START),))
            else:
                possible_time_start = set((
                    self._time_start.get_assumed_time(period)
                    for period in Period
                ))
            possible_start = set((
                self.user.timezone.localize(datetime.combine(
                    self._date_start.get_date(today),
                    time_start,
                ))
                for time_start in possible_time_start
            ))

        if self._duration:
            possible_end = set((
                start + self._duration
                for start in possible_start
            ))
        elif self._time_end:
            if self._date_end:  # Fixed date end
                possible_date_end = set((self._date_end.get_date(today),))
            else:  # Date end is either the same as date start or it is the day after
                possible_date_end = set((
                    start.date() + delta
                    for start in possible_start
                    for delta in (timedelta(), timedelta(days=1))
                ))
            possible_end = set((
                self.user.timezone.localize(datetime.combine(
                    date_end,
                    self._time_end.get_assumed_time(period),
                ))
                for date_end in possible_date_end
                for period in Period
            ))
        else:
            raise RuntimeError('One of _duration, _time_end must be set.')

        # Match all possible starts / ends to choose shortest possible event
        result_start, result_end = None, None
        for start, end in product(possible_start, possible_end):
            if end > start and (result_start is None or result_end - result_start > end - start):
                result_start, result_end = start, end

        if result_start is None or result_end is None:
            raise EventEndError('No possible end datetime found.')
        return result_start, result_end

    async def _handle_filling_slots(self, analytics: Analytics) -> ScenarioResult:
        assert self._state is CreateEventScenarioState.FILLING_SLOTS
        self._update_from_slots()
        missing_slot_request = self._request_missing_slot(analytics)  # walrus spot
        if missing_slot_request is not None:
            # Missing slot is found and needs to be requested.
            return missing_slot_request
        else:
            # Every slot is filled. Forming an event and asking user for confirmation.
            self._state = CreateEventScenarioState.AWAITING_CONFIRMATION
            try:
                datetime_start, datetime_end = self._get_event_start_end()
            except EventEndError:
                # User entered incorrect event start/end, finishing the scenario.
                return ScenarioResult(
                    response=ScenarioResponse(text=gettext('Could not find feasible event start / end time.')),
                    value=None,
                )

            today = timezone_today(self.user.timezone)
            if self._all_day:
                text = gettext('Create all day event "%(name)s" at %(date)s?') % dict(
                    name=self._event_name,
                    date=format_date(datetime_start, today),
                )
            elif datetime_start.date() == datetime_end.date():
                text = gettext('Create event "%(name)s" %(date)s from %(time_start)s to %(time_end)s?') % dict(
                    name=self._event_name,
                    date=format_date(datetime_start, today),
                    time_start=format_time(datetime_start),
                    time_end=format_time(datetime_end),
                )
            else:
                text = gettext('Create event "%(name)s" from %(datetime_start)s to %(datetime_end)s?') % dict(
                    name=self._event_name,
                    datetime_start=format_datetime(datetime_start, today),
                    datetime_end=format_datetime(datetime_end, today),
                )

            return self._build_request(
                text=text,
                requested_slot=(self._CONFIRMATION_SLOT, YesNo),
                contains_sensitive_data=True,
            )

    async def _handle_awaiting_confirmation(self, analytics: Analytics) -> ScenarioResult:
        assert self._state is CreateEventScenarioState.AWAITING_CONFIRMATION
        confirmation = self.require_slot(self._CONFIRMATION_SLOT, YesNo)
        if confirmation is YesNo.YES:
            analytics.actions.append(Action.EVENT_CREATED)

            datetime_start, datetime_end = self._get_event_start_end()
            if self._commit:
                assert self._event_name is not None
                event_id = await self.clients.public_calendar.create_event(
                    uid=self.user.uid,
                    user_ticket=self.user.user_ticket,
                    start_datetime=datetime_start,
                    end_datetime=datetime_end,
                    name=self._event_name,
                    all_day=self._all_day,
                )

                self.logger.context_push(event_id=event_id)
                self.logger.info('Event was created in calendar.')

            return ScenarioResult(
                response=ScenarioResponse(
                    text=gettext('Event created.'),
                    buttons=[get_calendar_day_button(datetime_start.date())],
                    commit=True,
                ),
                value=None,
            )
        else:
            return ScenarioResult(
                response=ScenarioResponse(text=gettext('Event creation cancelled.')),
                value=None,
            )

    async def handle(self) -> ScenarioResult:
        # Starter frame can be anything but the state will be INITIAL. Every subsequent frame is controlled by this
        # scenario and must always be CREATE_EVENT_SUBSEQUENT. Otherwise it means that user decided to restart this
        # scenario and we must reset all gathered slots.
        with scenario_event_create_time.labels(enum_value(self._state)).time:
            scenario_event_create_count.labels(enum_value(self._state)).inc()

            analytics = Analytics(intent=Intent.CREATE_EVENT)

            if not (
                self._state is CreateEventScenarioState.INITIAL
                or self._frame_name is FrameName.CREATE_EVENT_SUBSEQUENT
            ):
                raise CoreIrrelevantScenarioError('Create event scenario is being restarted.', analytics=analytics)

            # Next state after INITIAL is always FILLING_SLOTS.
            if self._state is CreateEventScenarioState.INITIAL:
                self._state = CreateEventScenarioState.FILLING_SLOTS

            if self._state is CreateEventScenarioState.FILLING_SLOTS:
                result = await self._handle_filling_slots(analytics)
            elif self._state is CreateEventScenarioState.AWAITING_CONFIRMATION:
                result = await self._handle_awaiting_confirmation(analytics)
            else:
                raise CoreIrrelevantScenarioError('Unexpected create event scenario state.', analytics=analytics)

            assert result.response
            result.response.analytics = analytics

            return result
