from enum import Enum, unique
from typing import Any

from mail.ciao.ciao.core.entities.enums import FrameName, YesNo
from mail.ciao.ciao.core.entities.scenario_response import ScenarioResponse
from mail.ciao.ciao.core.entities.scenario_result import ScenarioResult
from mail.ciao.ciao.core.exceptions import CoreIrrelevantScenarioError
from mail.ciao.ciao.core.scenarios.base import BaseScenario
from mail.ciao.ciao.interactions.calendar.entities import Event
from mail.ciao.ciao.interactions.calendar.exceptions import EventNotFoundCalendarError
from mail.ciao.ciao.utils.gettext import gettext


@unique
class DeleteEventScenarioState(Enum):
    INITIAL = 'initial'
    AWAITING_CONFIRMATION = 'awaiting_confirmation'


class DeleteEventScenario(BaseScenario[None]):
    """
    Accepts event, asks for confirmation, deletes event on commit.
    """

    scenario_name = 'delete_event'

    _SLOT_CONFIRMATION = 'delete_event_confirmation'

    def __init__(self,
                 *args: Any,
                 state: DeleteEventScenarioState = DeleteEventScenarioState.INITIAL,
                 event: Event,
                 **kwargs: Any,
                 ):
        super().__init__(*args, **kwargs)
        self._state = state
        self._event = event

    def _get_params(self) -> dict:
        return {
            'state': self._state,
            'event': self._event,
        }

    async def _handle_initial(self) -> ScenarioResult[None]:
        self._state = DeleteEventScenarioState.AWAITING_CONFIRMATION
        return ScenarioResult(
            response=ScenarioResponse(
                text=gettext('Delete event "%(name)s"?') % {'name': self._event.name},
                requested_slot=(self._SLOT_CONFIRMATION, YesNo),
                frame_name=FrameName.DELETE_EVENT,
                expected_frames=[FrameName.DELETE_EVENT],  # https://st.yandex-team.ru/DIALOG-6406
            )
        )

    async def _handle_awaiting_confirmation(self) -> ScenarioResult[None]:
        confirmation = self.require_slot(self._SLOT_CONFIRMATION, YesNo)
        if confirmation == YesNo.NO:
            return ScenarioResult(
                response=ScenarioResponse(
                    text=gettext('Event deletion cancelled.'),
                ),
                value=None,
            )
        else:
            if self._commit:
                try:
                    await self.clients.public_calendar.delete_event(
                        uid=self.user.uid,
                        user_ticket=self.user.user_ticket,
                        event_id=self._event.event_id,
                    )
                except EventNotFoundCalendarError:
                    pass
            return ScenarioResult(
                response=ScenarioResponse(
                    text=gettext('Event deleted.'),
                    commit=True,
                ),
                value=None,
            )

    async def handle(self) -> ScenarioResult[None]:
        if self._state == DeleteEventScenarioState.INITIAL:
            return await self._handle_initial()
        elif self._state == DeleteEventScenarioState.AWAITING_CONFIRMATION:
            return await self._handle_awaiting_confirmation()
        else:
            raise CoreIrrelevantScenarioError('Unexpected delete_event scenario state.')
