from datetime import date, datetime, timedelta
from enum import Enum, unique
from typing import Any, ClassVar, List, Optional, Tuple, cast

from pytz.tzinfo import BaseTzInfo

from sendr_utils import alist

from mail.ciao.ciao.core.entities.analytcs import Analytics, Intent
from mail.ciao.ciao.core.entities.button import TypeTextButton, get_calendar_day_button
from mail.ciao.ciao.core.entities.enums import DateEnum, EventListSingleEvent, FrameName, YesNo
from mail.ciao.ciao.core.entities.missing import MissingType
from mail.ciao.ciao.core.entities.scenario_response import ScenarioResponse
from mail.ciao.ciao.core.entities.scenario_result import ScenarioResult
from mail.ciao.ciao.core.exceptions import CoreIrrelevantScenarioError
from mail.ciao.ciao.core.scenarios.base import BaseScenario
from mail.ciao.ciao.core.scenarios.delete_event import DeleteEventScenario
from mail.ciao.ciao.core.scenarios.find_event import FindEventScenario
from mail.ciao.ciao.core.scenarios.reschedule_event import RescheduleEventScenario
from mail.ciao.ciao.interactions.calendar.entities import Event
from mail.ciao.ciao.utils.datetime import UserDate, is_today, is_tomorrow, timezone_today
from mail.ciao.ciao.utils.format import format_date, format_time
from mail.ciao.ciao.utils.gettext import gettext, ngettext


@unique
class EventListScenarioState(Enum):
    OVERVIEW = 'overview'
    AWAITING_YES_NO = 'awaiting_yes_no'
    AWAITING_ACTION = 'awaiting_action'
    DELETION = 'deletion'
    RESCHEDULING = 'rescheduling'


class EventListScenario(BaseScenario):
    """
    https://wiki.yandex-team.ru/mail/swat/ciao/#m-polucheniespiskasobytijj
    """

    scenario_name = 'event_list'

    _SLOT_DAY: ClassVar[str] = 'event_list_day'
    _SLOT_SINGLE_EVENT: ClassVar[str] = 'event_list_single_event'
    _SLOT_YES_NO_ANSWER: ClassVar[str] = 'event_list_yes_no_answer'

    _ANALYTICS = Analytics(intent=Intent.EVENT_LIST)

    def __init__(self,
                 *args: Any,
                 state: EventListScenarioState = EventListScenarioState.OVERVIEW,
                 events: Optional[List[Event]] = None,
                 day: Optional[date] = None,
                 found_event: Optional[Event] = None,
                 find_result: Optional[ScenarioResult[Optional[Event]]] = None,
                 delete_result: Optional[ScenarioResult[None]] = None,
                 reschedule_result: Optional[ScenarioResult[None]] = None,
                 **kwargs: Any,
                 ):
        super().__init__(*args, **kwargs)
        self._state = state
        self._events = events
        self._day = day
        self._found_event = cast(
            Optional[Event],
            find_result.value if find_result is not None else found_event
        )
        self._delete_result = delete_result
        self._reschedule_result = reschedule_result

    def _get_params(self) -> dict:
        return {
            'state': self._state,
            'events': self._events,
            'day': self._day,
            'found_event': self._found_event,
        }

    def _get_day(self) -> date:
        today = timezone_today(self.user.timezone)

        user_date = self.get_slot(self._SLOT_DAY, UserDate)
        if user_date is not MissingType.MISSING:
            return user_date.get_date(today)

        date_enum = self.get_slot(self._SLOT_DAY, DateEnum)
        if date_enum is not MissingType.MISSING:
            if date_enum is DateEnum.TODAY:
                return today
            elif date_enum is DateEnum.TOMORROW:
                return today + timedelta(days=1)
            else:
                raise RuntimeError('Unexpected DateEnum value.')

        return today

    @staticmethod
    def format_time_tz(dt: datetime, tz: BaseTzInfo) -> str:
        return format_time(dt.astimezone(tz))

    @classmethod
    def format_start_end(cls,
                         start: datetime,
                         end: datetime,
                         day: date,
                         tz: BaseTzInfo,
                         ) -> Tuple[str, str]:
        """
        Formats time interval while fitting it into day to be used in "from ... to ..." template.
        """
        day_start = tz.localize(datetime.combine(day, datetime.min.time()))
        day_end = day_start + timedelta(days=1)
        start_str = cls.format_time_tz(start, tz)
        end_str = cls.format_time_tz(end, tz)
        day_used = False
        if day_end <= end:
            end_str = gettext('end of day (genitive)')
            day_used = True
        if start <= day_start:
            start_str = gettext('beginning (genitive)')
            if not day_used:
                start_str += ' ' + gettext('day (genitive)')
        return start_str, end_str

    @classmethod
    def format_event(cls, event: Event, day: date, tz: BaseTzInfo) -> str:
        start, end = cls.format_start_end(event.start_ts, event.end_ts, day, tz)
        return gettext('Event "%(name)s" from %(start_time) to %(end_time)s') % dict(
            start_time=start,
            end_time=end,
            name=event.name,
        )

    def _events_list_result(self) -> ScenarioResult:
        assert self._events is not None and self._day is not None
        delete_text = gettext('Delete event. (suggest)')
        reschedule_text = gettext('Reschedule event. (suggest)')
        return ScenarioResult(
            response=ScenarioResponse(
                text='\n'.join((
                    self.format_event(event, self._day, self.user.timezone)
                    for event in self._events
                )),
                buttons=[get_calendar_day_button(self._day)],
                expected_frames=[FrameName.DELETE_EVENT, FrameName.RESCHEDULE_EVENT],
                contains_sensitive_data=True,
                suggests=[
                    TypeTextButton(title=delete_text, text=delete_text),
                    TypeTextButton(title=reschedule_text, text=reschedule_text),
                ],
            ),
        )

    async def _fetch_events(self) -> None:
        self._day = self._get_day()
        self._single_event = self.get_slot(self._SLOT_SINGLE_EVENT, EventListSingleEvent)

        # Request sanity check
        if self._single_event is EventListSingleEvent.NEXT and not is_today(self._day, self.user.timezone):
            raise CoreIrrelevantScenarioError('Requested next event of date other than current.',
                                              analytics=self._ANALYTICS)

        from_datetime = self.user.timezone.localize(datetime.combine(self._day, datetime.min.time()))
        to_datetime = from_datetime + timedelta(days=1)

        self._events = await alist(self.clients.public_calendar.get_events(
            uid=self.user.uid,
            user_ticket=self.user.user_ticket,
            from_datetime=from_datetime,
            to_datetime=to_datetime,
        ))
        assert self._events is not None

        # Filtering events if a single one requested
        if self._single_event is EventListSingleEvent.FIRST:
            self._events = self._events[:1]
        if self._single_event is EventListSingleEvent.NEXT:
            now = datetime.now(self.user.timezone)
            next_event = next((
                event
                for event in self._events
                if event.start_ts >= now
            ), None)
            self._events = [next_event] if next_event is not None else []

    async def _handle_overview(self) -> ScenarioResult:
        await self._fetch_events()
        assert self._day is not None and self._events is not None

        day_str = format_date(self._day, timezone_today(self.user.timezone))

        if not self._events:
            if is_today(self._day, self.user.timezone):
                text = gettext('You have no events today.')
            elif is_tomorrow(self._day, self.user.timezone):
                text = gettext('You have no events tomorrow.')
            else:
                text = gettext('You have no events at %(day)s.')

            return ScenarioResult(
                response=ScenarioResponse(
                    text=text % dict(day=day_str),
                    buttons=[get_calendar_day_button(self._day)],
                    contains_sensitive_data=True,
                ),
                value=None,
            )
        elif len(self._events) == 1:
            self._state = EventListScenarioState.AWAITING_ACTION
            return self._events_list_result()

        self._state = EventListScenarioState.AWAITING_YES_NO

        if is_today(self._day, self.user.timezone):
            text = ngettext(
                'You have 1 event today from %(start_time)s to %(end_time)s. Tell more?',
                'You have %(count)s events today from %(start_time)s to %(end_time)s. Tell more?',
                len(self._events),
            )
            speech = ngettext(
                'You have 1 event today from %(start_time)s to %(end_time)s. Tell more? (speech)',
                'You have %(count)s events today from %(start_time)s to %(end_time)s. Tell more? (speech)',
                len(self._events),
            )
        elif is_tomorrow(self._day, self.user.timezone):
            text = ngettext(
                'You have 1 event tomorrow from %(start_time)s to %(end_time)s. Tell more?',
                'You have %(count)s events tomorrow from %(start_time)s to %(end_time)s. Tell more?',
                len(self._events),
            )
            speech = ngettext(
                'You have 1 event tomorrow from %(start_time)s to %(end_time)s. Tell more? (speech)',
                'You have %(count)s events tomorrow from %(start_time)s to %(end_time)s. Tell more? (speech)',
                len(self._events),
            )
        else:
            text = ngettext(
                'You have 1 event at %(day)s from %(start_time)s to %(end_time)s. Tell more?',
                'You have %(count)s events at %(day)s from %(start_time)s to %(end_time)s. Tell more?',
                len(self._events),
            )
            speech = ngettext(
                'You have 1 event at %(day)s from %(start_time)s to %(end_time)s. Tell more? (speech)',
                'You have %(count)s events at %(day)s from %(start_time)s to %(end_time)s. Tell more? (speech)',
                len(self._events),
            )

        start_str, end_str = self.format_start_end(
            min(self._events, key=lambda e: e.start_ts).start_ts,
            max(self._events, key=lambda e: e.end_ts).end_ts,
            day=self._day,
            tz=self.user.timezone,
        )
        data = dict(
            day=day_str,
            start_time=start_str,
            end_time=end_str,
            count=len(self._events),
        )

        return ScenarioResult(
            response=ScenarioResponse(
                text=text % data,
                speech=speech % data,
                requested_slot=(self._SLOT_YES_NO_ANSWER, YesNo),
                contains_sensitive_data=True,
                frame_name=FrameName.EVENT_LIST_SUBSEQUENT,
            ),
        )

    async def _handle_yes_no_answer(self) -> ScenarioResult:
        yes_no_answer = self.require_slot(self._SLOT_YES_NO_ANSWER, YesNo)
        assert self._events, 'Must not get here unless there are events present.'
        assert self._day is not None
        if yes_no_answer == YesNo.YES:
            self._state = EventListScenarioState.AWAITING_ACTION
            return self._events_list_result()
        else:
            return ScenarioResult(
                response=ScenarioResponse(),
                value=None,
            )

    async def _handle_awaiting_action(self) -> ScenarioResult:
        if self._frame_name == FrameName.DELETE_EVENT:
            self._state = EventListScenarioState.DELETION
        elif self._frame_name == FrameName.RESCHEDULE_EVENT:
            self._state = EventListScenarioState.RESCHEDULING
        else:
            raise CoreIrrelevantScenarioError('Not an event_list action frame name.', analytics=self._ANALYTICS)
        assert self._events
        return ScenarioResult(
            call=(
                FindEventScenario(
                    events=self._events,
                    slots=self._slots,
                ),
                'find_result',
            ),
        )

    async def _handle_deletion(self) -> ScenarioResult:
        # Once deletion is complete, proxying it's response and finishing this scenario.
        if self._delete_result is not None:
            return ScenarioResult(
                response=self._delete_result.response,
                value=None,
            )
        elif self._found_event is None:
            return ScenarioResult(
                response=ScenarioResponse(
                    text=gettext('Event for deletion was not found.'),
                ),
                value=None,
            )
        else:
            return ScenarioResult(
                call=(
                    DeleteEventScenario(
                        frame_name=self._frame_name,
                        slots=self._slots,
                        event=self._found_event,
                    ),
                    'delete_result',
                )
            )

    async def _handle_rescheduling(self) -> ScenarioResult:
        if self._reschedule_result is not None:
            return ScenarioResult(
                response=self._reschedule_result.response,
                value=None,
            )
        elif self._found_event is None:
            return ScenarioResult(
                response=ScenarioResponse(text=gettext('Event for rescheduling was not found.')),
                value=None,
            )
        else:
            return ScenarioResult(
                call=(
                    RescheduleEventScenario(event=self._found_event),
                    'reschedule_result',
                ),
            )

    async def handle(self) -> ScenarioResult:
        if (
                self._state is not EventListScenarioState.OVERVIEW
                and self._frame_name is FrameName.EVENT_LIST
        ):
            raise CoreIrrelevantScenarioError('Event list scenario is being restarted.')

        self.logger.context_push(event_list_state=self._state)

        if self._state == EventListScenarioState.OVERVIEW:
            result = await self._handle_overview()
        elif self._state == EventListScenarioState.AWAITING_YES_NO:
            result = await self._handle_yes_no_answer()
        elif self._state == EventListScenarioState.AWAITING_ACTION:
            result = await self._handle_awaiting_action()
        elif self._state == EventListScenarioState.DELETION:
            result = await self._handle_deletion()
        elif self._state == EventListScenarioState.RESCHEDULING:
            result = await self._handle_rescheduling()
        else:
            raise CoreIrrelevantScenarioError('Unexpected list event scenario state.', analytics=self._ANALYTICS)

        if result.response:
            result.response.analytics = self._ANALYTICS
        return result
