from datetime import datetime, timedelta
from functools import partial
from typing import Any, List, Optional

import pylev
from pytz.tzinfo import BaseTzInfo

from mail.ciao.ciao.core.entities.enums import FrameName
from mail.ciao.ciao.core.entities.scenario_response import ScenarioResponse
from mail.ciao.ciao.core.entities.scenario_result import ScenarioResult
from mail.ciao.ciao.core.scenarios.base import BaseScenario
from mail.ciao.ciao.interactions.calendar.entities import Event
from mail.ciao.ciao.utils.datetime import Period, UserTime, time_distance
from mail.ciao.ciao.utils.gettext import gettext


class FindEventScenario(BaseScenario[Optional[Event]]):
    """
    Filters given list of events with one of passed filters. Requests filter from user in case none were passed.
    """

    scenario_name = 'find_event'

    _SLOT_EVENT_NAME = 'find_event_name'
    _SLOT_EVENT_TIME_START = 'find_event_time_start'

    def __init__(self,
                 *args: Any,
                 events: List[Event],
                 name: Optional[str] = None,
                 time_start: Optional[UserTime] = None,
                 **kwargs: Any,
                 ):
        super().__init__(*args, **kwargs)
        self._events = events
        self._name = self.get_slot(self._SLOT_EVENT_NAME, str, None) or name
        self._time_start = self.get_slot(self._SLOT_EVENT_TIME_START, UserTime, None) or time_start

    def _get_params(self) -> dict:
        return {
            'events': self._events,
            'name': self._name,
            'time_start': self._time_start,
        }

    @staticmethod
    def _name_key(name: str, e: Event) -> int:
        return pylev.levenshtein(name.lower(), e.name.lower())

    @staticmethod
    def _time_start_key(time_start: UserTime, tz: BaseTzInfo, e: Event) -> timedelta:
        if time_start.relative:
            now = datetime.now(tz)
            return abs(e.start_ts - time_start.get_absolute(now))
        else:
            return min((
                time_distance(
                    e.start_ts.astimezone(tz).time(),
                    time_start.get_assumed_time(period)
                )
                for period in Period
            ))

    async def handle(self) -> ScenarioResult[Optional[Event]]:
        event: Optional[Event] = None
        if len(self._events) == 1:
            event = self._events[0]
        elif self._name:
            event = min(self._events, key=partial(self._name_key, self._name))
        elif self._time_start:
            key = partial(self._time_start_key, self._time_start, self.user.timezone)
            event = min(self._events, key=key)
            # In case closest event start time is more than 1 hour off, assuming no match.
            if key(event) > timedelta(hours=1):
                event = None
        else:
            return ScenarioResult(
                response=ScenarioResponse(
                    text=gettext('Provide event name or start time.'),
                    expected_frames=[FrameName.FIND_EVENT],
                    expects_continuation=True,
                ),

            )
        return ScenarioResult(value=event)
