from datetime import datetime
from enum import Enum, unique
from typing import Any, Optional

from mail.ciao.ciao.conf import settings
from mail.ciao.ciao.core.entities.enums import FrameName, SysSlotType, YesNo
from mail.ciao.ciao.core.entities.scenario_response import RequestedSlotType, ScenarioResponse
from mail.ciao.ciao.core.entities.scenario_result import ScenarioResult
from mail.ciao.ciao.core.exceptions import CoreIrrelevantScenarioError
from mail.ciao.ciao.core.scenarios.base import BaseScenario
from mail.ciao.ciao.interactions.calendar.entities import Event
from mail.ciao.ciao.utils.datetime import UserDate, UserTime, timezone_today
from mail.ciao.ciao.utils.format import format_date, format_datetime, format_time
from mail.ciao.ciao.utils.gettext import gettext


@unique
class RescheduleEventScenarioState(Enum):
    INITIAL = 'initial'
    FILLING_SLOTS = 'filling_slots'
    AWAITING_CONFIRMATION = 'awaiting_confirmation'


class RescheduleEventScenario(BaseScenario):
    scenario_name = 'reschedule_event'

    _SLOT_NEW_DATE_START = 'reschedule_event_new_date_start'
    _SLOT_NEW_TIME_START = 'reschedule_event_new_time_start'
    _SLOT_CONFIRMATION = 'reschedule_event_confirmation'

    def __init__(self,
                 *args: Any,
                 event: Event,
                 state: RescheduleEventScenarioState = RescheduleEventScenarioState.INITIAL,
                 new_date_start: Optional[UserDate] = None,
                 new_time_start: Optional[UserTime] = None,
                 **kwargs: Any,
                 ):
        super().__init__(*args, **kwargs)
        self._state = state
        self._event = event
        self._new_date_start = self.get_slot(self._SLOT_NEW_DATE_START, UserDate, new_date_start)
        self._new_time_start = self.get_slot(self._SLOT_NEW_TIME_START, UserTime, new_time_start)

    def _get_params(self) -> dict:
        return {
            'state': self._state,
            'event': self._event,
            'new_date_start': self._new_date_start,
            'new_time_start': self._new_time_start,
        }

    def _get_new_datetime_start(self) -> datetime:
        assert self._new_date_start is not None
        if self._event.all_day:
            new_time_start = datetime.min.time()
        else:
            assert self._new_time_start is not None
            if self._new_time_start.relative:
                new_time_start = self._new_time_start.get_absolute(datetime.now(self.user.timezone)).time()
            else:
                new_time_start = self._new_time_start.get_fit_time(settings.WORK_DAY_START)
        return self.user.timezone.localize(datetime.combine(
            self._new_date_start.get_date(timezone_today(self.user.timezone)),
            new_time_start,
        ))

    async def _handle_filling_slots(self) -> ScenarioResult[None]:
        requested_slot: RequestedSlotType
        if self._new_date_start is None:
            text = gettext('What day to reschedule event?')
            requested_slot = (self._SLOT_NEW_DATE_START, SysSlotType.DATE)
        elif self._new_time_start is None and not self._event.all_day:
            text = gettext('What time to reschedule event?')
            requested_slot = (self._SLOT_NEW_TIME_START, SysSlotType.TIME)
        else:
            self._state = RescheduleEventScenarioState.AWAITING_CONFIRMATION
            old_datetime_start = self._event.start_ts.astimezone(self.user.timezone)
            new_datetime_start = self._get_new_datetime_start()
            if self._event.all_day:
                text = gettext('Reschedule "%(name)s" from %(date_start)s to %(new_date_start)s?')
            elif old_datetime_start.date() == new_datetime_start.date():
                text = gettext('Reschedule "%(name)s" from %(time_start)s to %(new_time_start)s?')
            else:
                text = gettext('Reschedule "%(name)s" from %(datetime_start)s to %(new_datetime_start)s?')
            today = timezone_today(self.user.timezone)
            text %= dict(
                name=self._event.name,
                time_start=format_time(old_datetime_start),
                new_time_start=format_time(new_datetime_start),
                date_start=format_date(old_datetime_start, today=today),
                new_date_start=format_date(new_datetime_start, today=today),
                datetime_start=format_datetime(old_datetime_start, today=today),
                new_datetime_start=format_datetime(new_datetime_start, today=today),
            )
            requested_slot = (self._SLOT_CONFIRMATION, YesNo)
        return ScenarioResult(
            response=ScenarioResponse(
                text=text,
                requested_slot=requested_slot,
                frame_name=FrameName.RESCHEDULE_EVENT,
                expected_frames=[FrameName.RESCHEDULE_EVENT],
            )
        )

    async def _handle_awaiting_confirmation(self) -> ScenarioResult[None]:
        confirmation = self.require_slot(self._SLOT_CONFIRMATION, YesNo)
        if confirmation is YesNo.NO:
            return ScenarioResult(
                response=ScenarioResponse(text=gettext('Event rescheduling cancelled.')),
                value=None,
            )
        else:
            if self._commit:
                new_datetime_start = self._get_new_datetime_start()
                await self.clients.public_calendar.update_event(
                    uid=self.user.uid,
                    user_ticket=self.user.user_ticket,
                    event_id=self._event.event_id,
                    start_ts=new_datetime_start,
                    end_ts=new_datetime_start + (self._event.end_ts - self._event.start_ts),
                    all_day=self._event.all_day,
                )
            return ScenarioResult(
                response=ScenarioResponse(
                    text=gettext('Event rescheduled.'),
                    commit=True,
                ),
                value=None,
            )

    async def handle(self) -> ScenarioResult[None]:
        if self._frame_name not in (None, FrameName.RESCHEDULE_EVENT):
            raise CoreIrrelevantScenarioError('Unexpected frame received.')

        if self._state is RescheduleEventScenarioState.INITIAL:
            self._state = RescheduleEventScenarioState.FILLING_SLOTS

        if self._state is RescheduleEventScenarioState.FILLING_SLOTS:
            return await self._handle_filling_slots()
        elif self._state is RescheduleEventScenarioState.AWAITING_CONFIRMATION:
            return await self._handle_awaiting_confirmation()
        else:
            raise CoreIrrelevantScenarioError('Unexpected state.')
