import logging
from typing import Dict, Optional, Type

from sendr_qlog import LoggerContext

from mail.ciao.ciao.interactions.base import BaseInteractionClient
from mail.ciao.ciao.interactions.calendar import PublicCalendarClient

default_logger = logging.getLogger('context_logger')


class InteractionClients:
    public_calendar: PublicCalendarClient

    _client_classes: Dict[str, Type[BaseInteractionClient]] = {
        'public_calendar': PublicCalendarClient,
    }

    def __init__(self, logger: Optional[LoggerContext] = None, request_id: Optional[str] = None):
        if logger is None:
            logger = LoggerContext(default_logger, {})
        self._logger = logger
        self._request_id = request_id
        self._clients: Dict[str, BaseInteractionClient] = {}

    def __getattr__(self, item):
        if item in self._client_classes:
            return self._get_client(item)
        raise AttributeError

    def _get_client(self, client: str) -> BaseInteractionClient:
        if client not in self._clients:
            self._clients[client] = self._client_classes[client](
                logger=self.logger,
                request_id=self.request_id,
            )
        return self._clients[client]

    @property
    def logger(self):
        return self._logger

    @property
    def request_id(self):
        return self._request_id

    async def close(self):
        for client in self._clients.values():
            await client.close()

    async def __aenter__(self):
        return self

    async def __aexit__(self, exc_type, exc_val, exc_tb):
        await self.close()
