import ssl
from types import SimpleNamespace
from typing import Any, TypeVar

from aiohttp import ClientResponse, TCPConnector, TraceConfig  # type: ignore

import sendr_interactions
from sendr_tvm import qloud_async_tvm as qtvm
from sendr_tvm.client import aiohttp as tvm_session

from mail.ciao.ciao.conf import settings
from mail.ciao.ciao.utils.stats import (
    interaction_method_response_status, interaction_method_response_time, interaction_response_status,
    interaction_response_time
)
from mail.ciao.ciao.utils.tvm import TVM_CONFIG

SSL_CONTEXT = ssl.create_default_context()
if settings.CA_FILE is not None:
    SSL_CONTEXT.load_verify_locations(cafile=settings.CA_FILE)


def create_connector():
    return TCPConnector(
        keepalive_timeout=settings.KEEPALIVE_TIMEOUT,
        limit=settings.CONNECTION_LIMIT,
        ssl=SSL_CONTEXT,
    )


ResponseType = TypeVar('ResponseType')
CiaoClientSession = tvm_session.sessions_producer(get_tvm=lambda: qtvm.QTVM(**TVM_CONFIG))


class BaseInteractionClient(sendr_interactions.AbstractInteractionClient[ResponseType]):
    DEBUG = settings.DEBUG
    REQUEST_RETRY_TIMEOUTS = settings.REQUEST_RETRY_TIMEOUTS
    TVM_SESSION_CLS = CiaoClientSession

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)

    @staticmethod
    async def on_connection_reuseconn(session: Any,
                                      trace_config_ctx: SimpleNamespace,
                                      params: Any,
                                      ) -> None:
        trace_config_ctx.trace_request_ctx.logger.info('Connection is being reused.')

    @staticmethod
    async def on_connection_create_start(session: Any,
                                         trace_config_ctx: SimpleNamespace,
                                         params: Any,
                                         ) -> None:
        trace_config_ctx.trace_request_ctx.logger.info('Connection is being created.')

    def _get_session_kwargs(self) -> dict:
        trace_config = TraceConfig()
        trace_config.on_connection_reuseconn.append(self.on_connection_reuseconn)
        trace_config.on_connection_create_start.append(self.on_connection_create_start)
        return {
            **super()._get_session_kwargs(),
            'trace_configs': [trace_config],
        }

    @classmethod
    def _response_time_metrics(cls, interaction_method: str, response_time: int) -> None:
        interaction_response_time.labels(cls.SERVICE).observe(response_time)
        interaction_method_response_time.labels(cls.SERVICE, interaction_method).observe(response_time)

    @classmethod
    def _response_status_metrics(cls, interaction_method: str, status: int) -> None:
        short_status = str(status // 100) + 'xx'
        interaction_response_status.labels(cls.SERVICE, short_status).inc()
        interaction_method_response_status.labels(cls.SERVICE, interaction_method, short_status).inc()

    def endpoint_url(self, relative_url, base_url_override=None):
        return f'{base_url_override or self.BASE_URL}/{relative_url}'

    @classmethod
    async def close_connector(cls):
        if cls.CONNECTOR:
            await cls.CONNECTOR.close()
            cls.CONNECTOR = None

    async def _make_request(self, *args, **kwargs) -> ClientResponse:
        return await super()._make_request(*args, trace_request_ctx=SimpleNamespace(logger=self.logger), **kwargs)
