import logging
import logging.config
from contextlib import contextmanager
from contextvars import ContextVar
from typing import Any, ClassVar

from sendr_qlog import LinesFormatter, LoggerContext, UniFormatter
from sendr_utils import temp_set

from mail.ciao.ciao.conf import settings


class SensitiveDataHolder:
    HIDE_SENSITIVE_DATA: ClassVar[bool] = settings.HIDE_SENSITIVE_DATA

    def __init__(self, obj: Any):
        self._obj = obj

    def __repr__(self):
        if self.HIDE_SENSITIVE_DATA:
            return '<sensitive_data>'
        return repr(self._obj)

    def __str__(self):
        if self.HIDE_SENSITIVE_DATA:
            return '<sensitive_data>'
        return str(self._obj)

    @classmethod
    @contextmanager
    def show_sensitive_data(cls):
        with temp_set(cls, 'HIDE_SENSITIVE_DATA', False):
            yield


class SensitiveDataFormatterMixin:
    def format(self, record):
        with SensitiveDataHolder.show_sensitive_data():
            return super().format(record)


class SensitiveDataLinesFormatter(SensitiveDataFormatterMixin, LinesFormatter):
    pass


class SensitiveDataUniFormatter(SensitiveDataFormatterMixin, UniFormatter):
    pass


def configure_logging():
    logging.config.dictConfig(settings.LOGGING)
    logging.root.setLevel(settings.LOG_LEVEL)


LOGGER: ContextVar[LoggerContext] = ContextVar('LOGGER')
