from logger import log
import time

TOKEN_TTL = 3600


class Cache:
    def __init__(self, path, min_ttl):
        self.path = path
        self.min_ttl = min_ttl
        self.active_tokens = self._load_active_tokens()

    def get(self, logins):
        ready = {l: self.active_tokens[l] for l in logins if l in self.active_tokens}
        not_ready = set(logins).difference(ready.keys())
        return ready, not_ready

    def _load_active_tokens(self):
        records = {}
        try:
            records = self._load_from_file()
        except Exception as e:
            log.error("cache failed to read data from file: {}".format(e))
        return self._filter_by_ttl(records)

    def _load_from_file(self):
        res = {}
        with open(self.path, "r") as file:
            while True:
                line = file.readline()
                if not line:
                    break
                parts = line.split(" ")
                if len(parts) != 3:
                    raise RuntimeError("parsing error: too few fields")
                res[parts[0]] = {"token": parts[1], "created_at": int(parts[2])}
        return res

    def _filter_by_ttl(self, tokens):
        res = {}
        now = int(time.time())
        for login, token_ts in tokens.items():
            ttl = TOKEN_TTL - (now - token_ts["created_at"])
            if ttl > self.min_ttl:
                res[login] = token_ts
        return res
