import dataclasses
from typing import Dict, Any, List, Type
import os
import yaml
from enum import Enum
import json


@dataclasses.dataclass
class BlackboxSettigns:
    base_url: str
    tvm_secret_file: str
    timeout: int = 5
    user_agent: str = "collectors-token-loader"


@dataclasses.dataclass
class TractorSettings:
    base_url: str


@dataclasses.dataclass
class DatabaseSettings:
    conninfo: str


@dataclasses.dataclass
class DecryptorSettings:
    key_versions_file: str


_RawSettings = Dict[str, Any]


class Settings:
    HIERARCHY_SEPARATOR = "/"

    def __init__(self) -> None:
        data = _load_config("TOKEN_LOADER_CONFIG_PATH")
        self.blackbox = _construct_from_dict(BlackboxSettigns, data.get("blackbox"))
        self.tractor = _construct_from_dict(TractorSettings, data.get("tractor"))
        self.ipa_db = _construct_from_dict(DatabaseSettings, data.get("ipa_db"))
        with open(data.get("key_versions_file"), "r") as file:
            self.key_versions = json.load(file)
        self.log_path = data.get(
            "log_path", "/var/log/collectors-ext/oauth_token_loader/loader.log"
        )
        self.validate_domain = data.get("validate_domain", True)
        self.min_token_ttl = data.get("min_token_ttl", 600)
        self.collectors_db = _get_shards_conninfo()


def _load_config(path: str) -> _RawSettings:
    config_path = os.environ.get(path)
    if config_path is None:
        raise RuntimeError('env variable "{}" not defined'.format(path))
    config = _load_config_file(config_path)
    _resolve_env_variables(config)
    return config


def _load_config_file(filename: str) -> _RawSettings:
    with open(filename) as config_file:
        config_data = yaml.safe_load(config_file)
        if "extends" not in config_data:
            return config_data
        parent_config = _load_config_file(config_data["extends"])
        return _merge_configs(parent_config, config_data)


def _resolve_env_variables(config: _RawSettings) -> None:
    updates = {}
    for key, val in config.items():
        if isinstance(val, dict):
            _resolve_env_variables(val)
        elif isinstance(val, str) and val.startswith("$"):
            updates[key] = os.environ[val[1:]]
    config.update(updates)


def _merge_configs(parent: _RawSettings, child: _RawSettings) -> _RawSettings:
    merged = dict()

    for key, parent_val in parent.items():
        if key in child:
            child_val = child[key]
            if type(parent_val) != type(child_val):
                raise Exception(
                    f"Bad override for {key} in child config(parent type is {type(parent_val)}, child type is {type(child_val)}"
                )

            if type(parent_val) is dict:
                merged[key] = _merge_configs(parent_val, child_val)
            elif type(parent_val) is list:
                merged[key] = parent_val + child_val
            else:
                merged[key] = child_val
        else:
            merged[key] = parent_val

    for key, child_val in child.items():
        if key not in parent:
            merged[key] = child_val

    return merged


def _construct_from_dict(dataclass: Type, data: Dict) -> Any:
    field_types = {f.name: f.type for f in dataclasses.fields(dataclass)}
    args = {}
    for k, v in data.items():
        if issubclass(field_types[k], Enum):
            args[k] = field_types[k][v.lower()]
        else:
            args[k] = field_types[k](v)
    return dataclass(**args)


def settings() -> Settings:
    if not hasattr(settings, "instance"):
        settings.instance = Settings()

    return settings.instance


def _get_shards_conninfo() -> List[str]:
    config = _load_config("SHARDS_CONFIG_PATH")["config"]

    conninfos = []
    for shard in config["shards"]:
        hosts = ",".join([host for host in shard["conninfo"]["hosts"]])
        params = shard["conninfo"]["params"]
        conninfos.append("host={} {}".format(hosts, params))

    return conninfos
