#!/usr/bin/python3
# -*- coding: UTF-8 -*-

from urllib.request import urlopen
import json
import psycopg2
import psycopg2.extras
import concurrent.futures
import re
import os
import yaml

CONFIG_FILE = "/etc/collectors-ext/config.yml"
INSTANCE_NAME = os.environ["QLOUD_INSTANCE"]


def get_pg_cursor(conninfo):
    conn = psycopg2.connect(conninfo)
    conn.autocommit = True
    cursor = conn.cursor(cursor_factory=psycopg2.extras.RealDictCursor)
    return cursor


def run_sql(sql, conninfo, monitoring_interval):
    cur = get_pg_cursor(conninfo)
    cur.execute(sql, (monitoring_interval,))
    return cur


def get_shards_conninfo(env_name):
    with open("/etc/collectors-ext/shards/{}.yml".format(env_name)) as file:
        config = yaml.safe_load(file)["config"]

    conninfos = []
    for shard in config["shards"]:
        hosts = ",".join([host for host in shard["conninfo"]["hosts"]])
        params = shard["conninfo"]["params"]
        conninfos.append("host={} {}".format(hosts, params))

    return conninfos


def run_sql_on_every_shard(sql, env_name, monitoring_interval):
    shards = get_shards_conninfo(env_name)
    with concurrent.futures.ThreadPoolExecutor(max_workers=len(shards)) as executor:
        futures = [
            executor.submit(run_sql, sql, conninfo, monitoring_interval) for conninfo in shards
        ]
        results = [future.result() for future in futures]
        return results


def update_old_collectors(file, env_name, monitoring_interval=3600):
    sql = """
    SELECT * FROM rpop.pop_profile
    WHERE
        to_timestamp((last_connect + (%s))::double precision) < now()
    AND
        is_on = 1
    AND
        bad_retries < 2
    AND
        last_connect > 0
    """
    old_collectors = {}
    for cursor in run_sql_on_every_shard(sql, env_name, monitoring_interval):
        for row in cursor:
            old_collectors[row["popid"]] = {
                "protocol": "pop3" if row["use_imap"] == 0 else "imap",
                "server": row["server"],
                "last_connect": row["last_connect"],
            }

    print("total old collectors: {}".format(len(old_collectors)))
    with open(file, "w") as f:
        json.dump(old_collectors, f)


def is_master():
    # Select master instance algorithm:
    # If we are first instance - we are master
    # If we are not first - check if first available
    # If first is not available - the second is master
    # There should NOT be more than 2 instances
    data = json.loads(urlopen("http://localhost:1/metadata").read().decode("utf-8"))
    names = [instance["name"] for instance in data["instances"]]
    names.sort()
    if INSTANCE_NAME == names[0]:
        return True

    instance = next(i for i in data["instances"] if i["name"] == names[0])
    try:
        url = "http://{}:11010/unistat".format(instance["fqdn"])
        res = urlopen(url, timeout=2).read().decode("utf-8")
    except:
        return True
    return False


if __name__ == "__main__":
    with open(CONFIG_FILE) as file:
        config = yaml.safe_load(file)

    master = is_master()
    print("current instance is {}".format("master" if master else "backup"), flush=True)
    if master:
        for env in config["envs"]:
            dump_file = "/var/collectors-ext/old_collectors_{}.json".format(env["name"])
            update_old_collectors(dump_file, env["name"], env["monitoring_interval"])
    elif os.path.exists(DUMP_FILE):
        os.remove(DUMP_FILE)
