#include <api/api_impl.h>
#include <common/util.h>
#include <common/idna.h>
#include <common/host_info.h>
#include <ymod_popclient/errors.h>
#include <ymod_imapclient/errors.h>
#include <ymod_webserver/server.h>
#include <yplatform/find.h>
#include <boost/algorithm/string.hpp>
#include <boost/property_tree/json_parser.hpp>
#include <boost/make_shared.hpp>

#include <api/http/ping_pong.h>
#include <api/http/list.h>
#include <api/http/create.h>
#include <api/http/edit.h>
#include <api/http/remove.h>
#include <api/http/enable.h>
#include <api/http/enable_abook.h>
#include <api/http/run_rpop.h>
#include <api/http/check_server.h>
#include <api/http/info.h>
#include <api/http/status.h>
#include <api/http/v2/check_login.h>
#include <api/http/v2/check_oauth.h>
#include <api/http/v2/setup_login.h>
#include <api/http/v2/setup_oauth.h>

#include <api/http/v2/ping.h>
#include <api/http/v2/get_smtp_data.h>

#include <api/cache.h>

#include <oauth/oauth.h>

namespace yrpopper::api {

template <class HandlerImplClass>
class WebServerHandler
{
    using StreamPtr = ymod_webserver::http::stream_ptr;

public:
    WebServerHandler(ApiImplPtr& api) : api(api)
    {
    }

    void set_log_prefix(PlatformContextPtr ctx)
    {
        auto log_prefix = ctx->uniq_id();
        ctx->logger().append_log_prefix(log_prefix);
    }

    template <typename... Args>
    void operator()(StreamPtr stream, Args&&... args)
    {
        set_log_prefix(stream->ctx());
        auto handler = std::make_shared<HandlerImplClass>(api);
        handler->setStream(stream);
        handler->exec(std::forward<Args>(args)...);
    }

private:
    ApiImplPtr api;
};

void api_impl::init(const yplatform::ptree& xml)
{
    typedef std::vector<yrpopper::common::ip_nets::net_description> net_desc_list;
    net_desc_list net_list;
    settings_.reset(new ApiSettings);
    settings_->my_owner_name = get_my_hostname();
    settings_->my_ip_addr = get_my_ipaddr();

    load_line_file(xml.get("yandex_nets", ""), "yandex nets", [&net_list](const std::string& net) {
        net_list.push_back(yrpopper::common::ip_nets::parse(net));
    });

    settings_->yandex_nets.set_net(net_list);

    load_line_file(
        xml.get("yandex_hosts", ""),
        "yandex hosts",
        boost::bind(&insert_string_2_set, _1, boost::ref(settings_->yandex_hosts)));

    load_line_file(
        xml.get("allowed_ip", ""),
        "allowed ip",
        boost::bind(&insert_string_2_set, _1, boost::ref(settings_->allowed_ip)));

    settings_->use_rc = xml.get("use_rc", true);
    if (settings_->use_rc)
    {
        rc_options_parser rc(settings_->rc);
        rc.parse_from_file(xml.get("rc_config", ""));
    }

    settings_->use_cache = xml.get("use_cache", true);
    settings_->log_extra = xml.get("log_extra", false);
    settings_->useFurita = xml.get("use_furita", true);

    std::string versioned_keys_file = xml.get("versioned_keys_file", "/etc/yrpop/versioned_keys");
    settings_->dkeys = compute_derived_keys(versioned_keys_file);

    auto secretPath = xml.get("smtp_secret_path", "/etc/yrpop/smtp.secret");
    std::ifstream secretStream(secretPath.c_str());
    if (!secretStream.is_open())
    {
        throw std::runtime_error("failed to open smtp_secret_path");
    }
    std::getline(secretStream, settings_->secret);

    boost::optional<const yplatform::ptree&> oauthHack = xml.get_child_optional("oauth_workaround");
    if (oauthHack)
    {
        for (auto& i : *oauthHack)
        {
            auto server = i.second.get<std::string>("<xmlattr>.server");
            auto literal = i.second.get("<xmlattr>.literal", "");
            settings_->oauth2Workaround.emplace_back(server, boost::lexical_cast<bool>(literal));
        }
    }

    auto forbidden_auth = xml.get_child_optional("forbidden_auth");
    if (forbidden_auth)
    {
        auto pop3_range = forbidden_auth->equal_range("pop3_welcome_messages");
        for (auto& [name, message] : boost::make_iterator_range(pop3_range))
        {
            settings_->forbidden_pop3_messages.push_back(message.data());
        }

        auto imap_range = forbidden_auth->equal_range("imap_id_responses");
        for (auto& [name, message] : boost::make_iterator_range(imap_range))
        {
            settings_->forbidden_imap_names.push_back(message.data());
        }
    }

    settings_->furita_host = xml.get("furita_url", "http://furita.mail.yandex.net/list.xml");
    settings_->furita_retries = xml.get("furita_retries", 3);

    settings_->sid = xml.get("blackbox_sid", 2u);

    std::string domains_config = xml.get("domain_settings", "/etc/yrpop/domain_settings.json");
    boost::property_tree::ptree json;
    boost::property_tree::read_json(domains_config, json);

    for (auto& i : json)
    {
        auto& node = i.second;

        server_param param;
        param.recieve.use_ssl = (node.get("useSSL", "true") == "true" ? true : false);

        auto optImap = node.get_child_optional("imap");
        if (!optImap)
        {
            auto pop = node.get_child("pop");
            param.use_imap = false;
            param.recieve.server = pop.get<string>("server");
            param.recieve.port = pop.get("port", (param.recieve.use_ssl ? 995 : 110));
        }
        else
        {
            auto imap = optImap.get();
            param.use_imap = true;
            param.recieve.server = imap.get<string>("server");
            param.recieve.port = imap.get("port", (param.recieve.use_ssl ? 993 : 143));
        }

        auto optSmtp = node.get_child_optional("smtp");
        if (optSmtp)
        {
            auto smtp = optSmtp.get();
            param.send.use_ssl = (smtp.get("useSSL", "true") == "true" ? true : false);
            param.send.server = smtp.get<string>("server");
            param.send.port = smtp.get("port", (param.send.use_ssl ? 465 : 25));
        }

        for (auto& domain : node.get_child("name"))
        {
            auto data = std::make_pair(domain.second.get_value<string>(), param);
            settings_->predefined_settings.insert(data);
        }
    }

    auto opt_hack = xml.get_child_optional("oauth_hack");
    if (opt_hack)
    {
        auto hack = opt_hack.get();
        OAuthHackSettings oauth_hack;
        oauth_hack.validate_email = hack.get("validate_email", true);
        settings_->oauth_hack_settings = oauth_hack;
    }
}

void api_impl::start()
{
    auto webServer = yplatform::find<ymod_webserver::server>("web_server");
    auto that = get_shared_from_this();

    webServer->bind("", { "/ping" }, WebServerHandler<http::PingPong>(that));
    webServer->bind("", { "/api/edit" }, WebServerHandler<http::Edit>(that));
    webServer->bind("", { "/api/create" }, WebServerHandler<http::Create>(that));
    webServer->bind("", { "/api/delete" }, WebServerHandler<http::Remove>(that));
    webServer->bind("", { "/api/enable" }, WebServerHandler<http::Enable>(that));
    webServer->bind("", { "/api/list" }, WebServerHandler<http::List>(that));
    webServer->bind("", { "/api/run" }, WebServerHandler<http::RunCollector>(that));
    webServer->bind("", { "/api/abook_enable" }, WebServerHandler<http::EnableAbook>(that));
    webServer->bind("", { "/api/check_server" }, WebServerHandler<http::CheckServer>(that));
    webServer->bind("", { "/api/info" }, WebServerHandler<http::Info>(that));
    webServer->bind("", { "/api/status" }, WebServerHandler<http::Status>(that));

    webServer->bind("", { "/api/v2/check_login" }, WebServerHandler<http::CheckLogin>(that));
    webServer->bind("", { "/api/v2/check_oauth" }, WebServerHandler<http::CheckOauth>(that));
    webServer->bind("", { "/api/v2/setup_login" }, WebServerHandler<http::SetupLogin>(that));
    webServer->bind("", { "/api/v2/setup_oauth" }, WebServerHandler<http::SetupOauth>(that));

    webServer->bind("https", { "/ping" }, WebServerHandler<http::PingPongV2>(that));
    webServer->bind("", { "/api/v2/smtp_data" }, WebServerHandler<http::GetSmtpData>(that));
}

void api_impl::handle_update(
    const yplatform::task_context_ptr& ctx,
    const std::string& /* mdb */,
    const std::string& suid,
    const popid_t& id)
{
    if (settings_->use_cache)
    {
        auto cacheService = yplatform::find<ymod_cache::cache>("cache_core");
        auto futureCache = cacheService->remove(ctx, pack(boost::make_tuple(suid, id)));
        futureCache.add_callback(
            boost::bind(&api_impl::handle_update_remove, this, ctx, suid, id, futureCache));
    }
}

void api_impl::handle_update_remove(
    const yplatform::task_context_ptr& ctx,
    const std::string& suid,
    const popid_t& id,
    ymod_cache::future_result fres)
{
    try
    {
        fres.get();
    }
    catch (std::exception& e)
    {
        TASK_LOG(ctx, error) << "/api/list cache error (remove): " << e.what();
    }
    catch (...)
    {
    }

    TASK_LOG(ctx, info) << "/api/list cache cleaned: suid=" << suid << ", popid=" << id;
    if (id != 0)
    {
        auto cacheService = yplatform::find<ymod_cache::cache>("cache_core");
        cacheService->remove(ctx, pack(boost::make_tuple(suid, 0)));
        TASK_LOG(ctx, info) << "/api/list cache cleaned: suid=" << suid << ", popid=0";
    }
}

} // namespace yrpopper::api

#include <yplatform/module_registration.h>

DEFINE_SERVICE_OBJECT(yrpopper::api::api_impl)
