#pragma once

#include <api/api.h>
#include <api/settings.h>
#include <db/interface_provider.h>
#include <ymod_popclient/call.h>
#include <ymod_imapclient/call.h>
#include <ymod_cache/cache.h>
#include <yplatform/util/atomic_count.h>
#include <api/rc/rc.h>
#include <oauth/oauth.h>

namespace yrpopper { namespace api {

typedef boost::shared_ptr<string> string_ptr_t;
typedef yplatform::future::future<string_ptr_t> future_string_ptr_t;
typedef yplatform::future::promise<string_ptr_t> promise_string_ptr_t;

using imap_client_ptr_t = ymod_imap_client::ImapClientPtr;

struct connect_result
{
    std::string server_ip;
};

using future_connect_result = yplatform::future::future<connect_result>;
using promise_connect_result = yplatform::future::promise<connect_result>;

struct connect_args
{
    connect_args(
        const yplatform::task_context_ptr& c,
        const string& m,
        const string& s,
        const string& ip,
        const task_info& t,
        boost::shared_ptr<rc_client> rc_email,
        boost::shared_ptr<rc_client> rc_ip,
        int retries)
        : ctx(c)
        , mdb(m)
        , suid(s)
        , ip(ip)
        , task(t)
        , rc_email(rc_email)
        , rc_ip(rc_ip)
        , retries(retries)
    {
    }

    yplatform::task_context_ptr ctx;
    string mdb;
    string suid;
    string ip;
    task_info task;
    boost::shared_ptr<rc_client> rc_email;
    boost::shared_ptr<rc_client> rc_ip;

    int retries;
};

struct connect_data
{
    connect_data(connect_args& a) : args(a)
    {
    }
    connect_args& args;
    imap_client_ptr_t imapClientInstance;
    promise_connect_result prom;
};
typedef boost::shared_ptr<connect_data> connect_d_ptr_t;

struct check_server_data
{
    check_server_data(const connect_args& a) : args(a)
    {
    }
    connect_args args;
    imap_client_ptr_t imapClientInstance;
    server_status result;
    promise_server_status prom;
};
typedef boost::shared_ptr<check_server_data> _cserver_d_ptr_t;

struct login_data
{
    login_data(const connect_args& a, imap_client_ptr_t imapClientInstance)
        : args(a), imapClientInstance(imapClientInstance), retries(a.retries)
    {
    }

    connect_args args;
    imap_client_ptr_t imapClientInstance;
    promise_bool_t prom;

    int retries;
};
typedef boost::shared_ptr<login_data> login_d_ptr_t;

struct check_dublicate_data
{
    enum result_state
    {
        ok,
        from_himself,
        dublicate,
        resolve_fail,
        list_fail,
        access_denied,
        blackbox_fail
    };

    yplatform::task_context_ptr ctx;
    string mdb;
    string suid;
    popid_t id = 0;
    string server;
    string login;
    task_info_list_ptr tasks;
    bool is_yandex_host = false;
    long resolve_size = 0;
    yplatform::util::atomic_count resolve_count = 0;
    yplatform::future::promise<result_state> prom;
};
typedef boost::shared_ptr<check_dublicate_data> check_dublicate_args;
typedef yplatform::future::future<check_dublicate_data::result_state> future_dublicate;

struct create_edit_data
{
    yplatform::task_context_ptr ctx;
    string mdb;
    string suid;
    string login;
    task_info task;
    promise_popid_t prom;

    bool reset_status;
};
typedef boost::shared_ptr<create_edit_data> create_args;
typedef boost::shared_ptr<create_edit_data> edit_args;

struct SmtpData
{
    std::string login;
    std::string pass;
    bool oauth;

    server_param connectData;
};
typedef yplatform::future::future<SmtpData> FutureSmtpData;
typedef yplatform::future::promise<SmtpData> PromiseSmtpData;

class api_impl;
typedef std::shared_ptr<api_impl> ApiImplPtr;

class api_impl : public api
{
public:
    const ApiSettings& settings() const
    {
        return *settings_;
    }

    virtual future_popid_t create(
        const yplatform::task_context_ptr& ctx,
        const string& mdb,
        const string& suid,
        const string& login,
        const task_info& task);

    virtual future_void_t remove(
        const yplatform::task_context_ptr& ctx,
        const string& mdb,
        const string& suid,
        const popid_t& id);

    virtual future_popid_t edit(
        const yplatform::task_context_ptr& ctx,
        const string& mdb,
        const string& suid,
        const string& login,
        const task_info& task,
        bool reset_status);

    virtual future_void_t enable(
        const yplatform::task_context_ptr& ctx,
        const string& mdb,
        const string& suid,
        const popid_t& id,
        bool on);

    virtual future_void_t enable_abook(
        const yplatform::task_context_ptr& ctx,
        const string& mdb,
        const string& suid,
        const popid_t& id);

    virtual future_task_info_list list(
        const yplatform::task_context_ptr& ctx,
        const string& mdb,
        const string& suid,
        const popid_t& id,
        bool skipCache);

    virtual future_string owner(const yplatform::task_context_ptr& ctx, const popid_t& id);

    virtual future_server_status check_server(
        const yplatform::task_context_ptr& ctx,
        const string& mdb,
        const string& suid,
        const std::string& ip,
        const task_info& task);

    virtual future_server_status check_auth(
        const yplatform::task_context_ptr& ctx,
        const task_info& task,
        const std::string& ip,
        int retries);

    virtual future_task_validity_t check_task_validity(
        const yplatform::task_context_ptr& ctx,
        const std::string& mdb,
        const string& suid,
        const base_task_info& task);

    virtual FutureSmtpData getSmtpData(
        const yplatform::task_context_ptr& ctx,
        const string& suid,
        const popid_t& id,
        const string& email);

    virtual FutureStatusData status(const yplatform::task_context_ptr& ctx, popid_t popid);
    virtual FutureStatusData hacked_status(const yplatform::task_context_ptr& ctx, popid_t popid);

    void init(const yplatform::ptree& xml);
    void start();

protected:
    ApiImplPtr get_shared_from_this()
    {
        return std::static_pointer_cast<api_impl>(shared_from_this());
    }

    void create_oauth_helper(
        const yplatform::task_context_ptr& ctx,
        const std::string& mdb,
        const std::string& suid,
        future_string fres,
        create_args args);

    void edit_oauth_helper(
        const yplatform::task_context_ptr& ctx,
        const std::string& mdb,
        const std::string& suid,
        future_string fres,
        edit_args args);

    future_task_info_list list_no_cache(
        const yplatform::task_context_ptr& ctx,
        const string& mdb,
        const string& suid,
        const popid_t& id);

    future_dublicate check_dublicate(
        const yplatform::task_context_ptr& ctx,
        const string& mdb,
        const string& suid,
        popid_t id,
        const string& server,
        const string& login);

    void handle_task_validity_check(
        const yplatform::task_context_ptr& ctx,
        future_dublicate fres,
        promise_task_validity_t prom);

    void handle_smtp_list(
        const yplatform::task_context_ptr& ctx,
        const string& email,
        popid_t popid,
        const string& suid,
        future_task_info_list futureList,
        PromiseSmtpData prom);

    void handle_list_actions(
        task_info_list_ptr taskList,
        FutureRpopActionsList futureActions,
        promise_task_info_list prom);

    void handle_list_no_cache(
        const yplatform::task_context_ptr& ctx,
        const string& mdb,
        const string& suid,
        future_task_info_list fres,
        promise_task_info_list prom);
    void handle_create_check(const create_args& args, future_dublicate res);
    void handle_edit_check(const edit_args& args, future_dublicate res);
    void handle_create_done(const create_args& args, future_popid_t res);
    void handle_edit_done(const edit_args& args, future_void_t res);
    void handle_enable_done(
        const yplatform::task_context_ptr& ctx,
        future_void_t res,
        promise_void_t prom,
        const std::string& mdb,
        const std::string& suid,
        popid_t& id);
    void handle_remove_done(
        const yplatform::task_context_ptr& ctx,
        promise_void_t prom,
        future_void_t res,
        const std::string& mdb,
        const std::string& suid,
        const popid_t& id);

    future_connect_result check_auth(connect_args args, settings_ptr settings);
    void handle_check_auth(const _cserver_d_ptr_t& d, future_connect_result fres);

    void handle_resolv_my(const check_dublicate_args& args, ymod_pop_client::future_string_ptr res);
    void handle_resolv_my_cont(
        const check_dublicate_args& args,
        yplatform::future::future<bool> fres);
    void handle_list_dub(const check_dublicate_args& args, future_task_info_list res);
    void handle_list_dub_yandex(
        const check_dublicate_args& args,
        yplatform::future::future<std::string> fres,
        task_info_list_ptr lst);
    void handle_resolv_dub_cont(
        const check_dublicate_args& args,
        yplatform::future::future<bool> fres,
        task_info i);
    void handle_resolv_dub(
        const check_dublicate_args& args,
        ymod_pop_client::future_string_ptr res,
        const string& login,
        task_info i);

    template <typename Result>
    void handle_result(
        yplatform::future::promise<Result> prom,
        yplatform::future::future<Result> res)
    {
        if (res.has_exception()) prom.set_exception(storage_error());
        else
            prom.set(res.get());
    }

    void list_cache_get(
        const yplatform::task_context_ptr& ctx,
        promise_task_info_list prom,
        ymod_cache::future_segment fres,
        const std::string& mdb,
        const std::string& suid,
        const popid_t& id);

    void list_cache_get_remove(
        const yplatform::task_context_ptr& ctx,
        promise_task_info_list prom,
        ymod_cache::future_result fres,
        const std::string& mdb,
        const std::string& suid,
        const popid_t& id);

    void list_cache_get_cont(
        const yplatform::task_context_ptr& ctx,
        promise_task_info_list prom,
        future_task_info_list fres,
        const std::string& mdb,
        const std::string& suid,
        const popid_t& id);

    void list_cache_set_tasks(
        const yplatform::task_context_ptr& ctx,
        task_info_list_ptr tasks,
        const std::string& suid,
        const popid_t& id,
        ymod_cache::future_result fres);

    void list_cache_set_task(
        const yplatform::task_context_ptr& ctx,
        task_info_list::const_iterator it,
        task_info_list_ptr tasks,
        const std::string& suid,
        ymod_cache::future_result fres);

    void handle_update(
        const yplatform::task_context_ptr& ctx,
        const std::string& mdb,
        const std::string& suid,
        const popid_t& id);

    void handle_update_remove(
        const yplatform::task_context_ptr& ctx,
        const std::string& suid,
        const popid_t& id,
        ymod_cache::future_result fres);

    _cserver_d_ptr_t prepareCheckData(
        const yplatform::task_context_ptr& ctx,
        const string& mdb,
        const string& suid,
        const std::string& ip,
        const task_info& task,
        int retries);

    void handle_status_data(
        const yplatform::task_context_ptr& ctx,
        popid_t popid,
        FutureStatusData futureStatus,
        PromiseStatusData prom);

    void handle_status_with_folders(
        future_imap_folders futureFolders,
        FutureStatusData futureStatus,
        PromiseStatusData prom);

    void handle_hacked_status_collector_info(
        const yplatform::task_context_ptr& ctx,
        popid_t popid,
        future_task_info_list collectorInfo,
        PromiseStatusData prom);

    settings_ptr settings_;
    db::ApiInterfacePtr dbInterface = db::InterfaceProvider::getApiInterface();
};

}}
