#pragma once

#include <common/task.h>
#include <yplatform/zerocopy/streambuf.h>
#include <boost/tuple/tuple.hpp>

namespace yrpopper { namespace api {

template <class V>
std::ostream& pack_impl(std::ostream& os, const V& v)
{
    os << v << std::endl;
    return os;
}

template <class A, class B>
std::ostream& pack_impl(std::ostream& os, const boost::tuple<A, B>& v)
{
    pack_impl(os, v.template get<0>());
    pack_impl(os, v.template get<1>());
    return os;
}

std::ostream& pack_impl(std::ostream& os, const yrpopper::task_info& t);

template <class Seq>
std::ostream& pack_seq(std::ostream& os, const Seq& v)
{
    pack_impl(os, v.size());
    for (const typename Seq::value_type& t : v)
        pack_impl(os, t);
    return os;
}

template <class T>
std::ostream& pack_impl(std::ostream& os, const std::vector<T>& v)
{
    return pack_seq(os, v);
}

template <class T>
std::ostream& pack_impl(std::ostream& os, const std::deque<T>& v)
{
    return pack_seq(os, v);
}

template <class T>
std::ostream& pack_impl(std::ostream& os, const std::list<T>& v)
{
    return pack_seq(os, v);
}

inline std::ostream& pack_impl(std::ostream& os, const yrpopper::task_info& t)
{
    pack_impl(os, t.popid);
    pack_impl(os, t.server);
    pack_impl(os, t.port);
    pack_impl(os, t.login);
    pack_impl(os, t.password);
    pack_impl(os, t.use_ssl);
    pack_impl(os, t.email);
    pack_impl(os, t.is_on);
    pack_impl(os, t.last_connect);
    pack_impl(os, t.session_duration);
    pack_impl(os, t.bad_retries);
    pack_impl(os, t.error_status);
    pack_impl(os, t.last_msg_count);
    pack_impl(os, t.abook_sync_state);
    pack_impl(os, t.mark_archive_read);
    pack_impl(os, t.action);
    pack_impl(os, t.use_imap);
    pack_impl(os, t.root_folder);
    pack_impl(os, t.leave_msgs);
    pack_impl(os, t.oauth_refresh_token);
    pack_impl(os, t.label_id);
    return os;
}

template <class V>
std::ostream& pack_impl(std::ostream& os, boost::shared_ptr<V> v)
{
    return pack_impl(os, *v);
}

template <class V>
inline yplatform::zerocopy::segment pack(const V& v)
{
    yplatform::zerocopy::streambuf buffer;
    std::ostream os(&buffer);
    pack_impl(os, v);
    os.flush();
    return buffer.detach(buffer.end());
}

template <class V>
V unpack(std::istream& is);

template <class V>
V unpack_impl(std::istream& is, V*)
{
    V v;
    if (is >> v)
    {
        is.ignore(std::numeric_limits<std::streamsize>::max(), '\n');
        return v;
    }
    throw std::runtime_error("unpack failed: unexpected end of stream");
}

inline std::string unpack_impl(std::istream& is, std::string*)
{
    std::string v;
    std::getline(is, v);
    return v;
}

template <class Seq>
Seq unpack_seq(std::istream& is)
{
    Seq res;
    int sz = unpack<int>(is);
    for (int i = 0; i < sz; ++i)
        res.push_back(unpack<typename Seq::value_type>(is));
    return res;
}

template <class V>
V unpack_impl(std::istream& is, std::vector<V>*)
{
    return unpack_seq<std::vector<V>>(is);
}

template <class V>
std::list<V> unpack_impl(std::istream& is, std::list<V>*)
{
    return unpack_seq<std::list<V>>(is);
}

template <class V>
std::deque<V> unpack_impl(std::istream& is, std::deque<V>*)
{
    return unpack_seq<std::deque<V>>(is);
}

template <class V>
inline boost::shared_ptr<V> unpack_impl(std::istream& is, boost::shared_ptr<V>*)
{
    return boost::shared_ptr<V>(new yrpopper::task_info_list(unpack<yrpopper::task_info_list>(is)));
}

inline yrpopper::task_info unpack_impl(std::istream& is, yrpopper::task_info*)
{
    yrpopper::task_info t;
    t.popid = unpack_impl(is, &t.popid);
    t.server = unpack_impl(is, &t.server);
    t.port = unpack_impl(is, &t.port);
    t.login = unpack_impl(is, &t.login);
    t.password = unpack_impl(is, &t.password);
    t.use_ssl = unpack_impl(is, &t.use_ssl);
    t.email = unpack_impl(is, &t.email);
    t.is_on = unpack_impl(is, &t.is_on);
    t.last_connect = unpack_impl(is, &t.last_connect);
    t.session_duration = unpack_impl(is, &t.session_duration);
    t.bad_retries = unpack_impl(is, &t.bad_retries);
    t.error_status = unpack_impl(is, &t.error_status);
    t.last_msg_count = unpack_impl(is, &t.last_msg_count);
    t.abook_sync_state = unpack_impl(is, &t.abook_sync_state);
    t.mark_archive_read = unpack_impl(is, &t.mark_archive_read);
    t.action = unpack_impl(is, &t.action);
    t.use_imap = unpack_impl(is, &t.use_imap);
    t.root_folder = unpack_impl(is, &t.root_folder);
    t.leave_msgs = unpack_impl(is, &t.leave_msgs);
    t.oauth_refresh_token = unpack_impl(is, &t.oauth_refresh_token);
    t.label_id = unpack_impl(is, &t.label_id);
    return t;
}

template <class V>
V unpack(std::istream& is)
{
    return unpack_impl(is, static_cast<V*>(0));
}

template <class V>
V unpack(const yplatform::zerocopy::segment& s)
{
    std::istringstream is(std::string(s.begin(), s.end()));
    return unpack<V>(is);
}

}}
