#pragma once

#include <api/http/common.h>
#include <api/util_blackbox.h>

namespace yrpopper { namespace api { namespace http {

class CheckServer : public HandlerImpl<server_status>
{
    typedef HandlerImpl<server_status> CheckServerHandlerBase;
    task_info task;

    std::string userIp;

public:
    CheckServer(const ApiImplPtr& api) : CheckServerHandlerBase(api, "check_server")
    {
    }

    bool process()
    {
        userIp = get_http_header_value(stream, "x-real-ip");
        apiParams.suid = get_http_param_value(stream, "suid");
        apiParams.mdb = get_http_param_value(stream, "mdb");

        try
        {
            task.popid = boost::lexical_cast<popid_t>(get_http_param_value(stream, "popid", "0"));
        }
        catch (...)
        {
            fail_finish(YRPOPPER_API_SYNTAX_ERROR_DESC, "can`t process request: bad popid value");
            return true;
        }

        task.password = get_http_param_value(stream, "password");
        auto social_task_id = get_http_param_value(stream, "social_task_id");
        if (task.password.empty() && social_task_id.empty())
        {
            fail_finish(YRPOPPER_API_SYNTAX_ERROR_DESC, "can`t process request: no credentials");
            return true;
        }

        task.login = get_http_param_value(stream, "login");
        if (task.login.empty() && social_task_id.empty())
        {
            fail_finish(YRPOPPER_API_SYNTAX_ERROR_DESC, "can`t process request: no login");
            return true;
        }

        task.email = get_http_param_value(stream, "email");
        if (task.email.empty())
        {
            task.email = task.login;
        }
        src_email = task.email;

        task.server = get_http_param_value(stream, "server");
        task.use_ssl = (get_http_param_value(stream, "ssl") == "1");
        task.use_imap = (get_http_param_value(stream, "imap") == "1");

        try
        {
            task.port = boost::lexical_cast<unsigned>(get_http_param_value(stream, "port", "0"));
        }
        catch (...)
        {
            fail_finish(YRPOPPER_API_SYNTAX_ERROR_DESC, "can`t process request: bad port value");
            return true;
        }

        if ((task.server.empty() || task.port == 0) && !boost::contains(task.email, "@") &&
            social_task_id.empty())
        {
            fail_finish(
                YRPOPPER_API_SYNTAX_ERROR_DESC,
                "wrong request: server information or email required ");
            return true;
        }

        if (!social_task_id.empty())
        {
            auto oauthModule = yplatform::find<oauth::OauthService>("oauth_module");
            auto futureRefreshToken = oauthModule->getRefreshToken(stream->ctx(), social_task_id);
            futureRefreshToken.add_callback(boost::bind(
                &CheckServer::handleRefreshToken, sharedAs<CheckServer>(), futureRefreshToken));
            return true;
        }

        tryAuthenticate();
        return true;
    }

    void handleRefreshToken(oauth::FutureRefreshTokenData futureRefreshToken)
    {
        try
        {
            auto data = futureRefreshToken.get();
            if (task.login.empty())
            {
                task.login = data.email;
                task.email = data.email;
                src_email = data.email;
            }

            if (!boost::algorithm::iequals(data.email, task.email))
            {
                return fail_finish(
                    YRPOPPER_API_SYNTAX_ERROR_DESC,
                    "can`t process request: social task email belongs to different user");
            }

            auto found = settings_.predefined_settings.find(data.provider);
            if (found == settings_.predefined_settings.end())
            {
                return fail_finish(
                    YRPOPPER_API_UNKNOWN_PROVIDER_ERROR_DESC,
                    "can`t process request: provider not found in config");
            }

            auto params = found->second;
            task.server = params.recieve.server;
            task.port = params.recieve.port;
            task.use_ssl = params.recieve.use_ssl;
            task.use_imap = true;

            task.oauth_refresh_token = data.refreshToken;
        }
        catch (const oauth::incorrect_data_exception& e)
        {
            return fail_finish(
                YRPOPPER_API_INCORRECT_OAUTH_DATA_ERROR_DESC,
                "can`t process request: incorrect oauth data(" + std::string(e.what()) + ")");
        }
        catch (const std::exception& e)
        {
            return fail_finish(
                YRPOPPER_API_SYNTAX_ERROR_DESC,
                "can`t process request: incorrect social_task_id(" + std::string(e.what()) + ")");
        }
        apiRequest();
    }

    void tryAuthenticate()
    {
        if (task.server.empty() || task.port == 0)
        {
            auto fres = isPddUser(stream->ctx(), task.email, settings_);
            fres.add_callback(
                boost::bind(&CheckServer::handleBlackbox, sharedAs<CheckServer>(), fres));
        }
        else
        {
            apiRequest();
        }
    }

protected:
    void handleBlackbox(FutureBoolResult fres)
    {
        if (fres.has_exception())
        {
            return fail_finish(
                YRPOPPER_API_INTERNAL_ERROR_DESC,
                "can't process request: " + get_exception_reason(fres));
        }

        auto isPddUser = fres.get();

        server_param param;
        if (isPddUser)
        {
            auto found = settings_.predefined_settings.find("yandex.ru");
            if (found != settings_.predefined_settings.end()) param = found->second;
        }
        else
        {
            size_t pos = task.email.find('@');
            string domain = task.email.substr(pos + 1);

            auto found = settings_.predefined_settings.find(domain);
            if (found != settings_.predefined_settings.end()) param = found->second;
        }

        if (param.recieve.server.empty())
        {
            fail_finish(
                YRPOPPER_API_UNKNOWN_DOMAIN_ERROR_DESC,
                "can`t process request: domain not found in config, full server info required");
            return;
        }

        task.server = param.recieve.server;
        task.port = param.recieve.port;
        task.use_imap = param.use_imap;
        task.use_ssl = param.recieve.use_ssl;

        apiRequest();
    }

    virtual FutureApiResult makeApiRequest()
    {
        return api_->check_server(stream->ctx(), apiParams.mdb, apiParams.suid, userIp, task);
    }

    virtual string logApiSuffix()
    {
        return string(", server='") + task.server + "', login=<" + task.login + ">";
    }

    virtual void respondeJson(json_builder& json, server_status& status)
    {
        json.begin_map()
            .value("request", request_name())
            .value("host", settings_.my_owner_name)
            .value("count", status.msg_count)
            .value("size", status.msg_size)
            .value("ip", status.ip)
            .value("id", uniq_id())
            .end_map();
    }

    virtual void respondeXml(xml_builder& xml, server_status& status)
    {
        xml.begin_node("yamail")
            .endl()
            .begin_node(request_name())
            .add_attr("host", settings_.my_owner_name)
            .add_attr("count", status.msg_count)
            .add_attr("size", status.msg_size)
            .add_attr("ip", status.ip)
            .add_attr("request_id", uniq_id())
            .end_node()
            .endl()
            .end_node();
    }
};

} // namespace http
} // namespace api
} // namespace yrpopper
