#pragma once

#include <api/api_impl.h>
#include <api/settings.h>
#include <api/http/api_params.h>

#include <db/interface_provider.h>

#include <common/typed_log.h>
#include <common/xml_buffer.h>
#include <common/json_buffer.h>
#include <common/util.h>
#include <yplatform/find.h>
#include <memory>
#include <boost/make_shared.hpp>

namespace yrpopper { namespace api { namespace http {

class ProxyRunHandler : public ymod_http_client::response_handler
{
public:
    ProxyRunHandler(const std::string& host) : host(host)
    {
    }

    void handle_data(const char* data, unsigned long long size)
    {
        body.append(data, size);
    }

    const std::string& getBody() const
    {
        return body;
    }

    const std::string& getHost() const
    {
        return host;
    }

private:
    std::string body;
    std::string host;
};

using ProxyRunHandlerPtr = boost::shared_ptr<ProxyRunHandler>;

struct ProxyResult
{
    std::string responseBody;
    std::string destinationHost;
};

typedef yplatform::future::future<ProxyResult> FutureProxyResult;
typedef yplatform::future::promise<ProxyResult> PromiseProxyResult;

class WebserverHandlerImpl : public std::enable_shared_from_this<WebserverHandlerImpl>
{
protected:
    using StreamPtr = ymod_webserver::http::stream_ptr;

public:
    WebserverHandlerImpl(const ApiImplPtr& api) : api_(api), settings(api->settings())
    {
    }

    virtual ~WebserverHandlerImpl(){};

    virtual const std::string& request_name() const = 0;

    void setStream(StreamPtr stream)
    {
        this->stream = stream;
    }

protected:
    template <class SiblingClass>
    std::shared_ptr<SiblingClass> sharedAs()
    {
        return std::dynamic_pointer_cast<SiblingClass>(shared_from_this());
    }

    std::string readRequestParam(std::string name, std::string def = "")
    {
        auto request = stream->request();
        auto const& params = request->url.params;
        auto found = request->url.params.find(name);
        if (found == params.end())
        {
            return def;
        }

        return found->second;
    }

    void finishError(
        ymod_webserver::codes::code code,
        const string& reason,
        const string& description)
    {
        json_builder json;
        json.begin_map()
            .begin_map("error")
            .value("method", request_name())
            .value("reason", reason)
            .value("description", description)
            .value("host", settings.my_owner_name)
            .value("request_id", uniq_id())
            .end_map()
            .end_map()
            .close();
        responseText(code, json.result());
    }

    void responseText(ymod_webserver::codes::code code, const string& text)
    {
        stream->result(code, text);
    }

    void response(
        ymod_webserver::codes::code code,
        const string& text,
        const std::string& contentType,
        const std::string& contentSubType)
    {
        stream->set_code(code);
        stream->set_content_type(contentType, contentSubType);
        stream->result_body(text);
    }

    string uniq_id()
    {
        return stream->ctx()->uniq_id();
    }

protected:
    StreamPtr stream;
    ApiImplPtr api_;
    const ApiSettings& settings;
};

class HandlerImplBase : public WebserverHandlerImpl
{
public:
    HandlerImplBase(const ApiImplPtr& api)
        : WebserverHandlerImpl(api)
        , settings_(api->settings())
        , dbInterface(db::InterfaceProvider::getApiInterface())
    {
    }

    virtual ~HandlerImplBase()
    {
    }

    virtual bool exec() = 0;
    virtual const string& request_name() const = 0;

protected:
    void fail_finish(const string& reason, const string& desc)
    {
        send_error_tag(reason, desc, settings_.my_owner_name);
    }

    template <typename Future>
    void fail_finish(Future fres)
    {
        try
        {
            fres.get();
        }
        catch (const ::yrpopper::api::error& e)
        {
            send_error_tag(e.public_message(), e.private_message(), settings_.my_owner_name);
            return;
        }
        catch (...)
        {
            send_error_tag(
                YRPOPPER_API_INTERNAL_ERROR_DESC,
                get_exception_reason(fres),
                settings_.my_owner_name);
            return;
        }

        send_error_tag(
            YRPOPPER_API_INTERNAL_ERROR_DESC,
            YRPOPPER_API_INTERNAL_ERROR_DESC,
            settings_.my_owner_name);
    }

    void send_error_tag(const string& reason, const string& desc, const string& host)
    {
        auto status = is_client_error(reason) ? "client_error"s : "error"s;
        typed_log::log_api(
            stream->ctx(),
            apiParams.uid,
            apiParams.suid,
            apiParams.popid,
            src_email,
            request_name(),
            status,
            desc);

        if (apiParams.use_json) send_error_tag_json(reason, desc, host);
        else
            send_error_tag_xml(reason, desc, host);

        stream->ctx()->custom_log_data["status"] = status;
        stream->ctx()->custom_log_data["error_reason"] = reason;
        L_(error) << uniq_id() << " end api: '" << request_name() << "', status=" << status
                  << ", reason='" << reason << "', desc='" << desc << "'";
    }

    void send_error_tag_json(const string& reason, const string& desc, const string& host)
    {
        json_builder builder;
        builder.begin_map()
            .begin_map("error")
            .value("method", request_name())
            .value("reason", reason)
            .value("description", desc)
            .value("host", host)
            .value("request_id", uniq_id())
            .end_map()
            .end_map()
            .close();

        response(ymod_webserver::codes::ok, builder.result(), "application", "json");
    }

    void send_error_tag_xml(const string& reason, const string& desc, const string& host)
    {
        xml_builder builder;
        builder.begin_node("yamail")
            .endl()
            .begin_node("error")
            .add_attr("method", request_name())
            .add_attr("reason", reason)
            .add_attr("description", desc)
            .add_attr("host", host)
            .add_attr("request_id", uniq_id())
            .end_node()
            .endl()
            .end_node();
        builder.close();

        response(ymod_webserver::codes::ok, builder.result(), "text", "xml");
    }

    bool is_client_error(const std::string& error)
    {
        static std::vector<std::string> client_errors = {
            YRPOPPER_API_SYNTAX_ERROR_DESC,
            YRPOPPER_API_INCORRECT_OAUTH_DATA_ERROR_DESC,
            YRPOPPER_API_RC_ERROR_DESC,
            YRPOPPER_API_CONNECT_ERROR_DESC,
            YRPOPPER_API_LOGIN_ERROR_DESC,
            YRPOPPER_API_SSL_ERROR_DESC,
            YRPOPPER_API_ACCESS_DENIED_ERROR_DESC,
            YRPOPPER_API_DUBLICATE_ERROR_DESC,
            YRPOPPER_API_HIMSELF_ERROR_DESC,
            YRPOPPER_API_NOT_FOUND_ERROR_DESC,
            YRPOPPER_API_FORBIDDEN_SERVER_ERROR_DESC
        };
        return std::find(client_errors.begin(), client_errors.end(), error) != client_errors.end();
    }

protected:
    ApiParams apiParams;
    const ApiSettings& settings_;
    db::ApiInterfacePtr dbInterface;
    std::string src_email;
};

template <typename ApiResultType, typename... RequiredParams>
class HandlerImpl : public HandlerImplBase
{
public:
    typedef yplatform::future::future<ApiResultType> FutureApiResult;
    typedef HandlerImpl<ApiResultType, RequiredParams...> This;

    HandlerImpl(const ApiImplPtr& api, const string& requestName)
        : HandlerImplBase(api), requestName(requestName)
    {
    }

    bool exec()
    {
        if (!apiParams.read<UseJson, Uid, RequiredParams...>(stream))
        {
            fail_finish(YRPOPPER_API_SYNTAX_ERROR_DESC, "can`t parse request");
            return true;
        }

        if (!checkParams()) return true;

        return process();
    }

    virtual bool checkParams()
    {
        return true;
    }

    virtual bool process()
    {
        apiRequest();
        return true;
    }

    virtual void apiRequest()
    {
        L_(info) << uniq_id() << " start api: '" << request_name() << ", popid=" << apiParams.popid
                 << this->logApiSuffix();
        if (needProxy())
        {
            auto proxyResultFuture = proxyRequest();
            proxyResultFuture.add_callback(
                boost::bind(&HandlerImpl::handleProxyResult, sharedAs<This>(), proxyResultFuture));
        }
        else
        {
            auto apiResultFuture = makeApiRequest();
            apiResultFuture.add_callback(
                boost::bind(&HandlerImpl::handleApiResult, sharedAs<This>(), apiResultFuture));
        }
    }

    virtual string logApiSuffix()
    {
        return "";
    }
    virtual FutureApiResult makeApiRequest() = 0;

    virtual bool needProxy()
    {
        return false;
    }
    virtual FutureProxyResult proxyRequest()
    {
        PromiseProxyResult prom;
        prom.set_exception(
            std::runtime_error("proxyRequest not implemented for " + request_name()));
        return prom;
    }

    void handleProxyResult(FutureProxyResult proxyResultFuture)
    {
        if (proxyResultFuture.has_exception())
        {
            fail_finish(proxyResultFuture);
            return;
        }

        auto proxyResult = proxyResultFuture.get();

        // req->headers["x-forwarded-for"] = proxyResult.destinationHost;
        response(
            ymod_webserver::codes::ok,
            proxyResult.responseBody,
            (apiParams.use_json ? "application" : "text"),
            (apiParams.use_json ? "json" : "xml"));

        L_(info) << uniq_id() << " end api: '" << request_name()
                 << "', status=proxy, destination=" << proxyResult.destinationHost;
    }

    void handleApiResult(FutureApiResult apiResultFuture)
    {
        if (apiResultFuture.has_exception())
        {
            fail_finish(apiResultFuture);
            return;
        }

        ApiResultType apiResult(std::move(apiResultFuture.get()));

        if (!checkApiResult(apiResult)) return;

        typed_log::log_api(
            stream->ctx(),
            apiParams.uid,
            apiParams.suid,
            apiParams.popid,
            src_email,
            request_name(),
            "success");

        stream->ctx()->custom_log_data["status"] = "ok";
        stream->ctx()->custom_log_data["error_reason"] = "ok";
        if (apiParams.use_json)
        {
            json_builder json;
            respondeJson(json, apiResult);
            json.close();

            response(ymod_webserver::codes::ok, json.result(), "application", "json");
        }
        else
        {
            xml_builder xml;
            respondeXml(xml, apiResult);
            xml.close();

            response(ymod_webserver::codes::ok, xml.result(), "text", "xml");
        }
        L_(info) << uniq_id() << " end api: '" << request_name() << "', status=success";
    }

    virtual bool checkApiResult(ApiResultType&)
    {
        return true;
    }

    virtual void respondeJson(json_builder& builder, ApiResultType& result) = 0;
    virtual void respondeXml(xml_builder& builder, ApiResultType& result) = 0;

protected:
    virtual const string& request_name() const
    {
        return requestName;
    }

protected:
    const string requestName;
};

} // namespace http
} // namespace api
} // namespace yrpopper
