#pragma once

#include <api/http/common.h>
#include <api/util_blackbox.h>
#include <oauth/hack/util.h>

namespace yrpopper { namespace api { namespace http {

class Create : public HandlerImpl<popid_t, Mdb, Suid, User>
{
    typedef HandlerImpl<popid_t, Mdb, Suid, User> CreateHandlerBase;
    task_info task;

public:
    Create(const ApiImplPtr& api) : CreateHandlerBase(api, "create")
    {
    }

    virtual bool process()
    {
        task.login = get_http_param_value(stream, "login");

        task.email = get_http_param_value(stream, "email");
        if (task.email.empty())
        {
            task.email = task.login;
        }
        src_email = task.email;

        task.server = get_http_param_value(stream, "server");
        task.use_ssl = (get_http_param_value(stream, "ssl") == "1");
        task.use_imap = (get_http_param_value(stream, "imap") == "1");

        task.password = get_http_param_value(stream, "password");
        if (!task.password.empty())
        {
            if (settings_.oauth_hack_settings &&
                oauth::hack::is_collector_hacked(task.server, task.password))
            {
                auto hack_settings = *settings_.oauth_hack_settings;
                if (!hack_settings.validate_email || oauth::hack::validate_email(task.login, apiParams.myLogin))
                {
                    // Non-empty oauth_refresh_token causes is_oauth=true.
                    // The value is used in oauth::hack::OauthServiceImpl.
                    auto fake_oauth_refresh_token = src_email;
                    task.oauth_refresh_token =
                        encrypt_password(fake_oauth_refresh_token, settings_.dkeys, apiParams.suid);
                    YLOG_G(info) << "new hacked oauth collector src_login: " << task.login
                                 << " dst_login: " << apiParams.myLogin;
                }
                else
                {
                    YLOG_G(info) << "hacked oauth collector email check failed for src_login: " << task.login
                                 << " dst_login: " << apiParams.myLogin;
                }
            }
            task.password = encrypt_password(task.password, settings_.dkeys, apiParams.suid);
        }

        auto social_task_id = get_http_param_value(stream, "social_task_id");
        if (task.password.empty() && social_task_id.empty())
        {
            fail_finish(YRPOPPER_API_SYNTAX_ERROR_DESC, "can`t process request: no credentials");
            return true;
        }

        if (task.login.empty() && social_task_id.empty())
        {
            fail_finish(YRPOPPER_API_SYNTAX_ERROR_DESC, "can`t process request: no login");
            return true;
        }

        try
        {
            task.port = boost::lexical_cast<unsigned>(get_http_param_value(stream, "port", "0"));
        }
        catch (...)
        {
            fail_finish(YRPOPPER_API_SYNTAX_ERROR_DESC, "can`t process request: bad port value");
            return true;
        }

        if ((task.server.empty() || task.port == 0) && !boost::contains(task.email, "@") &&
            social_task_id.empty())
        {
            fail_finish(
                YRPOPPER_API_SYNTAX_ERROR_DESC,
                "can`t process request: server information or email required ");
            return true;
        }

        task.leave_msgs = (get_http_param_value(stream, "no_delete_msgs") != "0");
        task.abook_sync_state = (get_http_param_value(stream, "sync_abook") == "1" ? 2 : 1);
        task.mark_archive_read = (get_http_param_value(stream, "mark_archive_read") == "1");
        task.root_folder = get_http_param_value(stream, "root_folder");
        task.label_id = get_http_param_value(stream, "label_id");

        if (!social_task_id.empty())
        {
            auto oauthModule = yplatform::find<oauth::OauthService>("oauth_module");
            auto futureRefreshToken = oauthModule->getRefreshToken(stream->ctx(), social_task_id);
            futureRefreshToken.add_callback(
                boost::bind(&Create::handleRefreshToken, sharedAs<Create>(), futureRefreshToken));
            return true;
        }

        tryAuthenticate();
        return true;
    }

    void handleRefreshToken(oauth::FutureRefreshTokenData futureRefreshToken)
    {
        try
        {
            auto data = futureRefreshToken.get();
            if (task.login.empty())
            {
                task.login = data.email;
                task.email = data.email;
                src_email = data.email;
            }
            if (!boost::algorithm::iequals(data.email, task.email))
            {
                return fail_finish(
                    YRPOPPER_API_SYNTAX_ERROR_DESC,
                    "can`t process request: social task email belongs to different user");
            }

            auto found = settings_.predefined_settings.find(data.provider);
            if (found == settings_.predefined_settings.end())
            {
                return fail_finish(
                    YRPOPPER_API_UNKNOWN_PROVIDER_ERROR_DESC,
                    "can`t process request: provider not found in config");
            }

            auto params = found->second;
            task.server = params.recieve.server;
            task.port = params.recieve.port;
            task.use_ssl = params.recieve.use_ssl;
            task.use_imap = true;

            task.oauth_refresh_token =
                encrypt_password(data.refreshToken, settings_.dkeys, apiParams.suid);
        }
        catch (const oauth::incorrect_data_exception& e)
        {
            return fail_finish(
                YRPOPPER_API_INCORRECT_OAUTH_DATA_ERROR_DESC,
                "can`t process request: incorrect oauth data(" + std::string(e.what()) + ")");
        }
        catch (const std::exception& e)
        {
            TASK_LOG(stream->ctx(), error) << "oauth exception: " << e.what();
            return fail_finish(
                YRPOPPER_API_SYNTAX_ERROR_DESC,
                "can`t process request: incorrect social_task_id(" + std::string(e.what()) + ")");
        }
        apiRequest();
    }

    void tryAuthenticate()
    {
        if (task.server.empty() || task.port == 0)
        {
            auto fres = isPddUser(stream->ctx(), task.email, settings_);
            fres.add_callback(boost::bind(&Create::handleBlackbox, sharedAs<Create>(), fres));
        }
        else
        {
            apiRequest();
        }
    }

protected:
    void handleBlackbox(FutureBoolResult fres)
    {
        if (fres.has_exception())
        {
            return fail_finish(
                YRPOPPER_API_INTERNAL_ERROR_DESC,
                "can`t process request: " + get_exception_reason(fres));
        }

        auto isPddUser = fres.get();

        server_param param;
        if (isPddUser)
        {
            auto found = settings_.predefined_settings.find("yandex.ru");
            if (found != settings_.predefined_settings.end()) param = found->second;
        }
        else
        {
            size_t pos = task.email.find('@');
            string domain = task.email.substr(pos + 1);

            auto found = settings_.predefined_settings.find(domain);
            if (found != settings_.predefined_settings.end()) param = found->second;
        }

        if (param.recieve.server.empty())
        {
            fail_finish(
                YRPOPPER_API_UNKNOWN_DOMAIN_ERROR_DESC,
                "can`t process request: domain not found in config, full server info required");
            return;
        }

        task.server = param.recieve.server;
        task.port = param.recieve.port;
        task.use_imap = param.use_imap;
        task.use_ssl = param.recieve.use_ssl;

        apiRequest();
    }

    virtual FutureApiResult makeApiRequest()
    {
        return api_->create(stream->ctx(), apiParams.mdb, apiParams.suid, apiParams.myLogin, task);
    }

    virtual string logApiSuffix()
    {
        return string(", server='") + task.server + "', login=<" + task.login + ">" +
            ", imap=" + boost::lexical_cast<string>(task.use_imap);
    }

    virtual void respondeJson(json_builder& json, popid_t& popid)
    {
        json.begin_map()
            .value("request", request_name())
            .value("host", settings_.my_owner_name)
            .value("id", uniq_id())
            .value("popid", boost::lexical_cast<std::string>(popid))
            .value("email", task.email)
            .end_map();
    }
    virtual void respondeXml(xml_builder& xml, popid_t& popid)
    {
        xml.begin_node("yamail")
            .endl()
            .begin_node(request_name())
            .add_attr("host", settings_.my_owner_name)
            .add_attr("request_id", uniq_id())
            .add_attr("popid", popid)
            .add_attr("email", task.email)
            .end_node()
            .endl()
            .end_node();
    }

};

} // namespace http
} // namespace api
} // namespace yrpopper
