#pragma once

#include <api/http/common.h>
#include <oauth/hack/util.h>

namespace yrpopper { namespace api { namespace http {

class Edit : public HandlerImpl<popid_t, Mdb, Suid, Popid, User>
{
    typedef HandlerImpl<popid_t, Mdb, Suid, Popid, User> EditHandlerBase;
    task_info task;

public:
    Edit(const ApiImplPtr& api) : EditHandlerBase(api, "edit")
    {
    }

protected:
    virtual bool process()
    {
        future_task_info_list fres =
            dbInterface->info(stream->ctx(), apiParams.suid, apiParams.popid);
        fres.add_callback(boost::bind(&Edit::handleTask, sharedAs<Edit>(), fres));
        return true;
    }

    void handleTask(future_task_info_list ftres)
    {
        if (ftres.has_exception())
        {
            fail_finish(ftres);
            return;
        }

        task_info_list_ptr task_list = ftres.get();
        if (task_list->size() != 1)
        {
            fail_finish(YRPOPPER_API_STORAGE_ERROR_DESC, "no task with such popid");
            return;
        }

        task = task_list->front();
        src_email = task.email;

        string social_task_id;
        try
        {
            string password = get_http_param_value(stream, "password");
            social_task_id = get_http_param_value(stream, "social_task_id");
            if (password.empty() && social_task_id.empty())
            {
                bool password_required = !get_http_param_value(stream, "server").empty();
                password_required |= !get_http_param_value(stream, "port").empty();
                password_required |= !get_http_param_value(stream, "ssl").empty();
                password_required |= !get_http_param_value(stream, "imap").empty();

                if (password_required)
                {
                    fail_finish(
                        YRPOPPER_API_SYNTAX_ERROR_DESC, "can`t process request: password missing");
                    return;
                }
            }
            else
            {
                auth_data_changed = true;

                task.server = get_http_param_value(stream, "server", task.server);

                if (!password.empty())
                {
                    if (settings_.oauth_hack_settings &&
                        oauth::hack::is_collector_hacked(task.server, password))
                    {
                        auto hack_settings = *settings_.oauth_hack_settings;
                        if (!hack_settings.validate_email || oauth::hack::validate_email(src_email, apiParams.myLogin))
                        {
                            // Non-empty oauth_refresh_token causes is_oauth=true.
                            // The value is used in oauth::hack::OauthServiceImpl.
                            auto fake_oauth_refresh_token = src_email;
                            task.oauth_refresh_token =
                                encrypt_password(fake_oauth_refresh_token, settings_.dkeys, apiParams.suid);
                            YLOG_G(info) << "edit hacked oauth collector src_login: " << src_email
                                        << " dst_login: " << apiParams.myLogin;
                        }
                        else
                        {
                            YLOG_G(info) << "hacked oauth collector check failed on edit: src_login=" << src_email
                                        << " dst_login=" << apiParams.myLogin;
                        }
                    }
                    task.password = encrypt_password(password, settings_.dkeys, apiParams.suid);
                }

                string port = get_http_param_value(stream, "port");
                if (!port.empty())
                {
                    task.port = boost::lexical_cast<unsigned>(port);
                }

                string ssl = get_http_param_value(stream, "ssl");
                if (!ssl.empty())
                {
                    task.use_ssl = (ssl != "0");
                }

                string imap = get_http_param_value(stream, "imap");
                if (!imap.empty())
                {
                    task.use_imap = (imap != "0");
                }
            }

            string no_delete_msgs = get_http_param_value(stream, "no_delete_msgs");
            if (!no_delete_msgs.empty())
            {
                task.leave_msgs = (no_delete_msgs != "0");
            }

            task.root_folder = get_http_param_value(stream, "root_folder", task.root_folder);
            task.label_id = get_http_param_value(stream, "label_id", task.label_id);
        }
        catch (...)
        {
            fail_finish(YRPOPPER_API_SYNTAX_ERROR_DESC, "can`t process request");
            return;
        }

        if (!social_task_id.empty())
        {
            auto oauthModule = yplatform::find<oauth::OauthService>("oauth_module");
            auto futureRefreshToken = oauthModule->getRefreshToken(stream->ctx(), social_task_id);
            futureRefreshToken.add_callback(
                boost::bind(&Edit::handleRefreshToken, sharedAs<Edit>(), futureRefreshToken));
            return;
        }

        apiRequest();
    }

    void handleRefreshToken(oauth::FutureRefreshTokenData futureRefreshToken)
    {
        try
        {
            auto data = futureRefreshToken.get();
            if (!boost::algorithm::iequals(data.email, task.email))
            {
                return fail_finish(
                    YRPOPPER_API_SYNTAX_ERROR_DESC,
                    "can`t process request: social task email belongs to different user");
            }
            task.oauth_refresh_token =
                encrypt_password(data.refreshToken, settings_.dkeys, apiParams.suid);
        }
        catch (const std::exception& e)
        {
            return fail_finish(
                YRPOPPER_API_SYNTAX_ERROR_DESC,
                "can`t process request: incorrect social_task_id(" + std::string(e.what()) + ")");
        }
        apiRequest();
    }

    virtual FutureApiResult makeApiRequest()
    {
        return api_->edit(
            stream->ctx(),
            apiParams.mdb,
            apiParams.suid,
            apiParams.myLogin,
            task,
            auth_data_changed);
    }

    virtual string logApiSuffix()
    {
        return ", imap=" + boost::lexical_cast<string>(task.use_imap);
    }

    virtual void respondeJson(json_builder& json, popid_t&)
    {
        json.begin_map()
            .value("request", request_name())
            .value("host", settings_.my_owner_name)
            .value("id", uniq_id())
            .end_map();
    }

    virtual void respondeXml(xml_builder& xml, popid_t&)
    {
        xml.begin_node("yamail")
            .endl()
            .begin_node(request_name())
            .add_attr("host", settings_.my_owner_name)
            .add_attr("request_id", uniq_id())
            .end_node()
            .endl()
            .end_node();
    }

    bool auth_data_changed = false;
};

} // namespace http
} // namespace api
} // namespace yrpopper
