#pragma once

#include <api/http/common.h>

namespace yrpopper { namespace api { namespace http {

class List : public HandlerImpl<task_info_list_ptr, Mdb, Suid>
{
    typedef HandlerImpl<task_info_list_ptr, Mdb, Suid> ListHandlerBase;
    bool infoRequest = false;
    bool skipCache = false;

public:
    List(const ApiImplPtr& api) : ListHandlerBase(api, "list")
    {
    }

protected:
    virtual bool checkParams()
    {
        // Popid are optional parameter for this request
        apiParams.popid = 0;
        skipCache = get_http_param_value(stream, "no_cache") == "1";
        try
        {
            apiParams.popid = boost::lexical_cast<popid_t>(get_http_param_value(stream, "popid"));
        }
        catch (...)
        {
        }
        return true;
    }

    virtual FutureApiResult makeApiRequest()
    {
        return api_->list(stream->ctx(), apiParams.mdb, apiParams.suid, apiParams.popid, skipCache);
    }

    virtual bool checkApiResult(task_info_list_ptr& list)
    {
        infoRequest = (apiParams.popid != 0);
        if (infoRequest && list->empty())
        {
            fail_finish(YRPOPPER_API_NOT_FOUND_ERROR_DESC, "rpop not found");
            return false;
        }
        return true;
    }

    virtual void respondeJson(json_builder& json, task_info_list_ptr& list)
    {
        json.begin_map()
            .value("request", request_name())
            .value("host", settings_.my_owner_name)
            .value("id", uniq_id())
            .begin_array("rpops");

        for (auto& item : *list)
        {
            bool isOauth = item.oauth_refresh_token.size();
            json.begin_map()
                .value("popid", boost::lexical_cast<std::string>(item.popid))
                .value("server", item.server)
                .value("port", boost::lexical_cast<std::string>(item.port))
                .value("login", item.login)
                .value("use_ssl", item.use_ssl)
                .value("email", item.email)
                .value("is_on", boost::lexical_cast<std::string>(item.is_on))
                .value("last_connect", boost::lexical_cast<std::string>(item.last_connect))
                .value("session_duration", boost::lexical_cast<std::string>(item.session_duration))
                .value("bad_retries", item.bad_retries)
                .value("error_status", item.error_status)
                .value("last_msg_count", item.last_msg_count)
                .value("leave_msgs", item.leave_msgs)
                .value("abook_sync_state", boost::lexical_cast<std::string>(item.abook_sync_state))
                .value("imap", item.use_imap)
                .value("root_folder", item.root_folder)
                .value("label_id", item.label_id)
                .value("is_oauth", isOauth);

            if (isOauth)
            {
                auto oauthModule = yplatform::find<oauth::OauthService>("oauth_module");
                json.value("oauth_app", oauthModule->getOauthApplication(item.server));
            }

            if (infoRequest)
            {
                json.value("password", "").value("mark_archive_read", item.mark_archive_read);
            }

            if (item.action.size() > 0)
            {
                json.begin_array("actions");
                for (auto& action : item.action)
                {
                    json.array_value(boost::lexical_cast<std::string>(action));
                }
                json.end_array();
            }
            json.end_map();
        }

        json.end_array().end_map();
    }

    virtual void respondeXml(xml_builder& xml, task_info_list_ptr& list)
    {
        xml.begin_node("yamail")
            .endl()
            .begin_node(request_name())
            .add_attr("host", settings_.my_owner_name)
            .add_attr("count", list->size())
            .add_attr("request_id", uniq_id())
            .endl();

        for (auto& item : *list)
        {
            bool isOauth = item.oauth_refresh_token.size();
            xml.begin_node("rpop")
                .add_attr("popid", item.popid)
                .add_attr("server", item.server)
                .add_attr("port", item.port)
                .add_attr("login", item.login)
                .add_attr("use_ssl", (item.use_ssl ? 1 : 0))
                .add_attr("email", item.email)
                .add_attr("is_on", item.is_on)
                .add_attr("last_connect", item.last_connect)
                .add_attr("session_duration", item.session_duration)
                .add_attr("bad_retries", item.bad_retries)
                .add_attr("error_status", item.error_status)
                .add_attr("last_msg_count", item.last_msg_count)
                .add_attr("leave_msgs", (item.leave_msgs ? 1 : 0))
                .add_attr("abook_sync_state", item.abook_sync_state)
                .add_attr("imap", item.use_imap)
                .add_attr("root_folder", item.root_folder)
                .add_attr("label_id", item.label_id)
                .add_attr("is_oauth", isOauth);

            if (isOauth)
            {
                auto oauthModule = yplatform::find<oauth::OauthService>("oauth_module");
                xml.add_attr("oauth_app", oauthModule->getOauthApplication(item.server));
            }

            if (infoRequest)
            {
                xml.add_attr("password", item.password)
                    .add_attr("mark_archive_read", item.mark_archive_read ? 1 : 0);
            }

            for (auto& action : item.action)
            {
                xml.begin_node("action").print_body(action).end_node();
            }

            xml.end_node().endl();
        }

        xml.end_node().end_node();
    }
};

} // namespace http
} // namespace api
} // namespace yrpopper
