#pragma once

#include <api/http/common.h>
#include <api/error.h>
#include <scheduler/scheduler.h>

namespace yrpopper { namespace api { namespace http {

class RunCollector : public HandlerImpl<VoidResult, Popid>
{
    typedef HandlerImpl<VoidResult, Popid> RunCollectorHandlerBase;
    bool verbose = false;
    bool clean = false;
    bool processor_ng = false;
    bool synced = false;

    std::string collectorOwner;

public:
    RunCollector(const ApiImplPtr& api) : RunCollectorHandlerBase(api, "run")
    {
    }

    virtual bool process()
    {
        try
        {
            clean = (get_http_param_value(stream, "clean") == "yes");
            verbose = (get_http_param_value(stream, "verbose") == "yes");
            processor_ng = (get_http_param_value(stream, "processor_ng") == "yes");
            synced = (get_http_param_value(stream, "synced") == "yes");
        }
        catch (...)
        {
        }

        if (clean) verbose = true;

        auto futureOwner = api_->owner(stream->ctx(), apiParams.popid);
        auto self = sharedAs<RunCollector>();
        futureOwner.add_callback([=]() { self->handleCollectorOwner(futureOwner); });
        return true;
    }

    virtual bool needProxy()
    {
        return collectorOwner != settings_.my_owner_name;
    }

    virtual FutureProxyResult proxyRequest()
    {
        PromiseProxyResult prom;

        auto httpClient = yplatform::find<yhttp::call>("http_client");
        ymod_http_client::remote_point_info_ptr ownerRemoteInfo;
        try
        {
            ownerRemoteInfo = httpClient->make_rm_info("https://" + collectorOwner, "");
        }
        catch (...)
        {
            prom.set_exception(owner_invalid_error());
            return prom;
        }

        auto httpHandler = boost::make_shared<ProxyRunHandler>(collectorOwner);
        auto httpFuture = httpClient->get_url(
            stream->ctx(),
            httpHandler,
            ownerRemoteInfo,
            "api/run?popid=" + boost::lexical_cast<std::string>(apiParams.popid) +
                (verbose ? "&verbose=yes" : "") + (clean ? "&clean=yes" : "") +
                (processor_ng ? "&processor_ng=yes" : "") + (synced ? "&synced=yes" : "") +
                (apiParams.use_json ? "&json=1" : "") + "&_=" + stream->ctx()->uniq_id());
        auto self = sharedAs<RunCollector>();
        httpFuture.add_callback([=]() { self->handleProxyRequest(httpFuture, httpHandler, prom); });

        return prom;
    }

    virtual FutureApiResult makeApiRequest()
    {
        auto schedulerService = yplatform::find<scheduler::SchedulerService>("rpop_scheduler_pq");
        std::set<scheduler::TaskRunParams::RunFlag> flags;

        if (clean) flags.insert(scheduler::TaskRunParams::RunFlag::Clean);
        if (verbose) flags.insert(scheduler::TaskRunParams::RunFlag::Verbose);
        if (processor_ng) flags.insert(scheduler::TaskRunParams::RunFlag::ProcessorNG);
        if (synced) flags.insert(scheduler::TaskRunParams::RunFlag::Syncronized);

        if (apiParams.use_json)
        {
            flags.insert(scheduler::TaskRunParams::RunFlag::ProxyJSON);
        }
        else
        {
            flags.insert(scheduler::TaskRunParams::RunFlag::ProxyXML);
        }

        auto params = std::make_shared<scheduler::TaskRunParams>(flags);
        return schedulerService->forceTaskRun(stream->ctx(), apiParams.popid, params);
    }

protected:
    void handleCollectorOwner(future_string futureOwner)
    {
        if (futureOwner.has_exception())
        {
            fail_finish(futureOwner);
            return;
        }

        collectorOwner = futureOwner.get();
        if (collectorOwner.empty())
        {
            fail_finish("owner null error", YRPOPPER_API_OWNER_NULL_ERROR_DESC);
            return;
        }
        apiRequest();
    }

    void handleProxyRequest(
        ymod_http_client::future_void_t httpFuture,
        ProxyRunHandlerPtr httpHandler,
        PromiseProxyResult prom)
    {
        try
        {
            httpFuture.get();
        }
        catch (...)
        {
            prom.set_current_exception();
        }
        ProxyResult res;
        res.destinationHost = httpHandler->getHost();
        res.responseBody = httpHandler->getBody();

        prom.set(res);
    }

    virtual void respondeJson(json_builder& json, VoidResult&)
    {
        json.begin_map()
            .value("request", request_name())
            .value("host", settings_.my_owner_name)
            .value("id", uniq_id())
            .end_map();
    }

    virtual void respondeXml(xml_builder& xml, VoidResult&)
    {
        xml.begin_node("yamail")
            .endl()
            .begin_node(request_name())
            .add_attr("host", settings_.my_owner_name)
            .add_attr("request_id", uniq_id())
            .end_node()
            .endl()
            .end_node();
    }
};

} // namespace http
} // namespace api
} // namespace yrpopper
