#pragma once

#include <api/http/common.h>

namespace yrpopper { namespace api { namespace http {

class CheckLogin : public HandlerImpl<server_status>
{
    typedef HandlerImpl<server_status> CheckLoginHandlerBase;
    task_info task;
    std::string userIp;
    int retries = 2;

public:
    CheckLogin(const ApiImplPtr& api) : CheckLoginHandlerBase(api, "check_login")
    {
    }

    bool process()
    {
        userIp = get_http_header_value(stream, "x-real-ip");
        task.email = get_http_param_value(stream, "email");
        task.login = task.email;
        task.password = get_http_param_value(stream, "password");

        task.server = get_http_param_value(stream, "server");

        task.use_ssl = (get_http_param_value(stream, "ssl", "1") == "1");
        task.use_imap = (get_http_param_value(stream, "imap", "1") == "1");

        try
        {
            task.port = boost::lexical_cast<unsigned>(get_http_param_value(stream, "port", "0"));
            retries = boost::lexical_cast<unsigned>(get_http_param_value(stream, "retries", "2"));
        }
        catch (...)
        {
            fail_finish(
                YRPOPPER_API_SYNTAX_ERROR_DESC,
                "can`t process request: port has non-integer value");
            return true;
        }

        try
        {
            retries = boost::lexical_cast<unsigned>(get_http_param_value(stream, "retries", "2"));
        }
        catch (...)
        {
            fail_finish(
                YRPOPPER_API_SYNTAX_ERROR_DESC,
                "can`t process request: retries has non-integer value");
            return true;
        }

        if (task.email.empty())
        {
            fail_finish(YRPOPPER_API_SYNTAX_ERROR_DESC, "can`t process request: no email");
            return true;
        }

        if (task.password.empty())
        {
            fail_finish(YRPOPPER_API_SYNTAX_ERROR_DESC, "can`t process request: no credentials");
            return true;
        }

        if (task.server.empty() || task.port == 0)
        {
            size_t pos = task.email.find('@');
            if (pos == task.email.npos)
            {
                fail_finish(
                    YRPOPPER_API_SYNTAX_ERROR_DESC,
                    "can`t process request: wrong email " + task.email);
                return true;
            }
            string domain = task.email.substr(pos + 1);

            server_param param;
            auto found = settings_.predefined_settings.find(domain);
            if (found != settings_.predefined_settings.end()) param = found->second;

            if (param.recieve.server.empty())
            {
                fail_finish(
                    YRPOPPER_API_UNKNOWN_DOMAIN_ERROR_DESC,
                    "server not found in config: " + domain + ", full server info required");
                return true;
            }

            task.server = param.recieve.server;
            task.port = param.recieve.port;
            task.use_ssl = param.recieve.use_ssl;
            task.use_imap = param.use_imap;
        }

        apiRequest();
        return true;
    }

protected:
    virtual FutureApiResult makeApiRequest()
    {
        return api_->check_auth(stream->ctx(), task, userIp, retries);
    }

    virtual string logApiSuffix()
    {
        return string(", server='") + task.server + "', login=<" + task.login + ">";
    }

    virtual void respondeJson(json_builder& json, server_status& status)
    {
        json.begin_map()
            .value("request", request_name())
            .value("host", settings_.my_owner_name)
            .value("ip", status.ip)
            .value("email", task.email)
            .value("id", uniq_id())
            .end_map();
    }

    virtual void respondeXml(xml_builder& xml, server_status& status)
    {
        xml.begin_node("yamail")
            .endl()
            .begin_node(request_name())
            .add_attr("host", settings_.my_owner_name)
            .add_attr("ip", status.ip)
            .add_attr("email", task.email)
            .add_attr("request_id", uniq_id())
            .end_node()
            .endl()
            .end_node();
    }
};

} // namespace http
} // namespace api
} // namespace yrpopper
