#pragma once

#include <api/http/common.h>

namespace yrpopper { namespace api { namespace http {

class CheckOauth : public HandlerImpl<server_status>
{
    typedef HandlerImpl<server_status> CheckOauthHandlerBase;
    task_info task;
    std::string userIp;
    int retries = 2;

public:
    CheckOauth(const ApiImplPtr& api) : CheckOauthHandlerBase(api, "check_oauth")
    {
    }

    bool process()
    {
        userIp = get_http_header_value(stream, "x-real-ip");
        std::string social_task_id = get_http_param_value(stream, "social_task_id");
        try
        {
            retries = boost::lexical_cast<unsigned>(get_http_param_value(stream, "retries", "2"));
        }
        catch (...)
        {
            fail_finish(
                YRPOPPER_API_SYNTAX_ERROR_DESC,
                "can`t process request: retries has non-integer value");
            return true;
        }

        if (social_task_id.empty())
        {
            fail_finish(YRPOPPER_API_SYNTAX_ERROR_DESC, "can`t process request: no credentials");
            return true;
        }

        auto oauthModule = yplatform::find<oauth::OauthService>("oauth_module");
        auto futureRefreshToken = oauthModule->getRefreshToken(stream->ctx(), social_task_id);
        futureRefreshToken.add_callback(boost::bind(
            &CheckOauth::handleRefreshToken, sharedAs<CheckOauth>(), futureRefreshToken));
        return true;
    }

    void handleRefreshToken(oauth::FutureRefreshTokenData futureRefreshToken)
    {
        try
        {
            auto data = futureRefreshToken.get();
            task.oauth_refresh_token = data.refreshToken;
            task.login = data.email;
            task.email = data.email;

            server_param param;
            auto found = settings_.predefined_settings.find(data.provider);
            if (found != settings_.predefined_settings.end()) param = found->second;

            if (param.recieve.server.empty())
            {
                return fail_finish(
                    YRPOPPER_API_UNKNOWN_DOMAIN_ERROR_DESC,
                    "server not found in config: " + task.server);
            }

            task.server = param.recieve.server;
            task.port = param.recieve.port;
            task.use_imap = param.use_imap;
            task.use_ssl = param.recieve.use_ssl;
        }
        catch (const std::exception& e)
        {
            return fail_finish(
                YRPOPPER_API_SYNTAX_ERROR_DESC,
                "can`t process request: incorrect social_task_id(" + std::string(e.what()) + ")");
        }
        apiRequest();
    }

protected:
    virtual FutureApiResult makeApiRequest()
    {
        return api_->check_auth(stream->ctx(), task, userIp, retries);
    }

    virtual string logApiSuffix()
    {
        return string(", server='") + task.server + "', login=<" + task.login + ">";
    }

    virtual void respondeJson(json_builder& json, server_status& status)
    {
        json.begin_map()
            .value("request", request_name())
            .value("host", settings_.my_owner_name)
            .value("ip", status.ip)
            .value("email", task.email)
            .value("id", uniq_id())
            .end_map();
    }

    virtual void respondeXml(xml_builder& xml, server_status& status)
    {
        xml.begin_node("yamail")
            .endl()
            .begin_node(request_name())
            .add_attr("host", settings_.my_owner_name)
            .add_attr("ip", status.ip)
            .add_attr("email", task.email)
            .add_attr("request_id", uniq_id())
            .end_node()
            .endl()
            .end_node();
    }
};

} // namespace http
} // namespace api
} // namespace yrpopper
