#pragma once

#include <api/http/common.h>
#include <api/util_blackbox.h>
#include <common/json_builder.h>
#include <string>

namespace yrpopper { namespace api { namespace http {

class GetSmtpData : public WebserverHandlerImpl
{
public:
    GetSmtpData(const ApiImplPtr& api) : WebserverHandlerImpl(api)
    {
    }

    void exec()
    {
        auto uid = readRequestParam("uid");
        if (uid.empty())
        {
            return finishError(
                ymod_webserver::codes::bad_request, "wrong request params", "empty uid");
        }

        try
        {
            popid = boost::lexical_cast<popid_t>(readRequestParam("popid", "0"));
        }
        catch (...)
        {
            return finishError(
                ymod_webserver::codes::bad_request, "wrong request params", "bad popid");
        }

        email = readRequestParam("email");
        if (email.empty() && !popid)
        {
            return finishError(
                ymod_webserver::codes::bad_request,
                "wrong request params",
                "empty email and popid");
        }

        auto secret = readRequestParam("secret");
        if (secret != settings.secret || secret.empty())
        {
            return finishError(
                ymod_webserver::codes::bad_request, "wrong request params", "wrong secret");
        }

        auto futureRes = getSuidMdb(stream->ctx(), uid, settings);
        auto self = sharedAs<GetSmtpData>();
        futureRes.add_callback([=]() { self->handleBlackbox(futureRes); });
    }

    const std::string& request_name() const override
    {
        static const std::string name = "smtp_data";
        return name;
    };

private:
    void handleBlackbox(FutureSuidMdb fres)
    {
        if (fres.has_exception())
        {
            return finishError(
                ymod_webserver::codes::internal_server_error,
                "blackbox error",
                get_exception_reason(fres));
        }

        auto suidMdb = fres.get();
        auto result = api_->getSmtpData(stream->ctx(), suidMdb->suid, popid, email);
        auto self = sharedAs<GetSmtpData>();
        result.add_callback([=]() { self->handleSmtpData(result); });
    }

    void handleSmtpData(FutureSmtpData futureSmtp)
    {
        if (futureSmtp.has_exception())
        {
            return finishError(
                ymod_webserver::codes::internal_server_error,
                "internal api error",
                get_exception_reason(futureSmtp));
        }

        auto smtpData = futureSmtp.get();

        json_builder json;
        json.begin_map()
            .value("request", std::string(request_name()))
            .value("host", settings.my_owner_name)
            .value("id", uniq_id())
            .begin_map("smtp_data")
            .value("login", smtpData.login)
            .value("pass", smtpData.pass)
            .value("oauth", smtpData.oauth)
            .value("server", smtpData.connectData.send.server)
            .value("port", boost::lexical_cast<string>(smtpData.connectData.send.port))
            .value("ssl", (smtpData.connectData.send.use_ssl ? "1" : "0"))
            .end_map()
            .end_map()
            .close();
        responseText(ymod_webserver::codes::ok, json.result());
    }

private:
    popid_t popid;
    string email;
};

} // namespace http
} // namespace api
} // namespace yrpopper
