#pragma once

#include <api/http/common.h>
#include <api/util_blackbox.h>

namespace yrpopper { namespace api { namespace http {

class SetupCollector : public HandlerImpl<popid_t>
{
    typedef HandlerImpl<popid_t> SetupCollectorHandlerBase;
    bool taskFound = false;

public:
    SetupCollector(const ApiImplPtr& api, const string& requestName)
        : SetupCollectorHandlerBase(api, requestName)
    {
    }

protected:
    virtual bool process()
    {
        std::string uid = get_http_param_value(stream, "uid");
        if (uid.empty())
        {
            fail_finish(YRPOPPER_API_SYNTAX_ERROR_DESC, "can`t process request: no uid");
            return true;
        }

        auto futureRes = getSuidMdb(stream->ctx(), uid, settings_);
        futureRes.add_callback(
            boost::bind(&SetupCollector::handleBlackbox, sharedAs<SetupCollector>(), futureRes));
        return true;
    }

    void handleBlackbox(FutureSuidMdb fres)
    {
        if (fres.has_exception())
        {
            return fail_finish(fres);
        }

        auto suidMdb = fres.get();
        apiParams.mdb = suidMdb->mdb;
        apiParams.suid = suidMdb->suid;

        prepareParams();
    }

    virtual void prepareParams() = 0;
    virtual std::string getEmail() = 0;
    virtual void handleTask(const task_info& task) = 0;
    virtual task_info_ptr getTask() = 0;

    virtual void handleParams()
    {
        future_task_info_list fres = dbInterface->list(stream->ctx(), apiParams.suid);
        fres.add_callback(
            boost::bind(&SetupCollector::handleTaskList, sharedAs<SetupCollector>(), fres));
    }

    void handleTaskList(future_task_info_list ftres)
    {
        if (ftres.has_exception())
        {
            fail_finish(ftres);
            return;
        }

        task_info_list_ptr task_list = ftres.get();
        for (auto& task : *task_list)
        {
            if (boost::algorithm::iequals(task.email, getEmail()))
            {
                handleTask(task);
                taskFound = true;
                break;
            }
        }
        apiRequest();
    }

    virtual FutureApiResult makeApiRequest()
    {
        if (taskFound)
        {
            return api_->edit(
                stream->ctx(), apiParams.mdb, apiParams.suid, apiParams.myLogin, *getTask(), true);
        }
        return api_->create(
            stream->ctx(), apiParams.mdb, apiParams.suid, apiParams.myLogin, *getTask());
    }

    virtual string logApiSuffix()
    {
        return ", imap=" + boost::lexical_cast<string>(getTask()->use_imap);
    }

    virtual void respondeJson(json_builder& json, popid_t& popid)
    {
        json.begin_map()
            .value("request", request_name())
            .value("host", settings_.my_owner_name)
            .value("id", uniq_id())
            .value("email", getEmail())
            .value("popid", boost::lexical_cast<std::string>(popid))
            .end_map();
    }

    virtual void respondeXml(xml_builder& xml, popid_t& popid)
    {
        xml.begin_node("yamail")
            .endl()
            .begin_node(request_name())
            .add_attr("host", settings_.my_owner_name)
            .add_attr("request_id", uniq_id())
            .add_attr("email", getEmail())
            .add_attr("popid", popid)
            .end_node()
            .endl()
            .end_node();
    }
};

} // namespace http
} // namespace api
} // namespace yrpopper
