#pragma once

#include <api/http/v2/setup_collector.h>

namespace yrpopper { namespace api { namespace http {

class SetupOauth : public SetupCollector
{
    oauth::RefreshTokenData oauthData;
    task_info_ptr taskInfo;
    server_param serverParam;

public:
    SetupOauth(const ApiImplPtr& api)
        : SetupCollector(api, "setup_oauth"), taskInfo(std::make_shared<task_info>())
    {
    }

protected:
    virtual void prepareParams()
    {
        auto task_id = get_http_param_value(stream, "social_task_id");
        auto oauthModule = yplatform::find<oauth::OauthService>("oauth_module");
        auto futureRefreshToken = oauthModule->getRefreshToken(stream->ctx(), task_id);
        futureRefreshToken.add_callback(boost::bind(
            &SetupOauth::handleRefreshToken, sharedAs<SetupOauth>(), futureRefreshToken));
    }

    virtual void handleTask(const task_info& task)
    {
        taskInfo = std::make_shared<task_info>(task);
        setTaskParams();
    }

    virtual std::string getEmail()
    {
        return oauthData.email;
    }

    virtual task_info_ptr getTask()
    {
        return taskInfo;
    }

private:
    void handleRefreshToken(oauth::FutureRefreshTokenData futureRefreshToken)
    {
        try
        {
            oauthData = futureRefreshToken.get();

            auto found = settings_.predefined_settings.find(oauthData.provider);
            if (found != settings_.predefined_settings.end()) serverParam = found->second;

            if (serverParam.recieve.server.empty())
            {
                return fail_finish(
                    YRPOPPER_API_UNKNOWN_DOMAIN_ERROR_DESC,
                    "provider not found in config: " + oauthData.provider);
            }

            setTaskParams();
        }
        catch (const std::exception& e)
        {
            return fail_finish(
                YRPOPPER_API_SYNTAX_ERROR_DESC,
                "can`t process request: incorrect social_task_id(" + std::string(e.what()) + ")");
        }
        handleParams();
    }

    void setTaskParams()
    {
        taskInfo->password = "";
        taskInfo->oauth_refresh_token =
            encrypt_password(oauthData.refreshToken, settings_.dkeys, apiParams.suid);
        taskInfo->login = oauthData.email;
        taskInfo->email = oauthData.email;

        taskInfo->server = serverParam.recieve.server;
        taskInfo->port = serverParam.recieve.port;
        taskInfo->use_imap = serverParam.use_imap;
        taskInfo->use_ssl = serverParam.recieve.use_ssl;
    }
};

} // namespace http
} // namespace api
} // namespace yrpopper
