#include "external_auth/imap_auth_op.h"
#include "external_auth/pop3_auth_op.h"

#include <api/api_impl.h>
#include <oauth/oauth.h>

#include <yplatform/find.h>

namespace yrpopper::api {

_cserver_d_ptr_t api_impl::prepareCheckData(
    const yplatform::task_context_ptr& ctx,
    const std::string& mdb,
    const std::string& suid,
    const std::string& ip,
    const task_info& task,
    int retries)
{
    boost::shared_ptr<rc_client> rc_email;
    boost::shared_ptr<rc_client> rc_ip;
    if (settings_->use_rc && !task.login.empty())
    {
        boost::asio::io_service* ios = yplatform::global_net_reactor->io();
        rc_email.reset(new rc_client(*ios, settings_->rc));
        if (!ip.empty())
        {
            rc_ip.reset(new rc_client(*ios, settings_->rc));
        }
    }

    return boost::make_shared<check_server_data>(
        connect_args(ctx, mdb, suid, ip, task, rc_email, rc_ip, retries));
}

future_server_status api_impl::check_auth(
    const yplatform::task_context_ptr& ctx,
    const task_info& task,
    const std::string& ip,
    int retries)
{
    auto checkData = prepareCheckData(ctx, "", "", ip, task, retries);
    auto futureConnect = check_auth(checkData->args, settings_);
    futureConnect.add_callback(boost::bind(
        &api_impl::handle_check_auth, yplatform::shared_from(this), checkData, futureConnect));
    return checkData->prom;
}

future_server_status api_impl::check_server(
    const yplatform::task_context_ptr& ctx,
    const std::string& mdb,
    const std::string& suid,
    const std::string& ip,
    const task_info& task)
{
    auto checkData = prepareCheckData(ctx, mdb, suid, ip, task, 1);
    auto futureConnect = check_auth(checkData->args, settings_);
    futureConnect.add_callback(boost::bind(
        &api_impl::handle_check_auth, yplatform::shared_from(this), checkData, futureConnect));
    return checkData->prom;
}

void api_impl::handle_check_auth(const _cserver_d_ptr_t& checkData, future_connect_result res)
{
    try
    {
        auto connectRes = res.get();
        checkData->result.ip = connectRes.server_ip;
        checkData->prom.set(checkData->result);
    }
    catch (...)
    {
        checkData->prom.set_current_exception();
        return;
    }
}

future_task_validity_t api_impl::check_task_validity(
    const yplatform::task_context_ptr& ctx,
    const std::string& mdb,
    const string& suid,
    const base_task_info& task)
{
    promise_task_validity_t prom;
    auto futureCheck = check_dublicate(ctx, mdb, suid, task.popid, task.server, task.login);
    auto checkCallback =
        boost::bind(&api_impl::handle_task_validity_check, this, ctx, futureCheck, prom);
    futureCheck.add_callback(checkCallback);
    return prom;
}

void api_impl::handle_task_validity_check(
    const yplatform::task_context_ptr& ctx,
    future_dublicate fres,
    promise_task_validity_t prom)
{
    try
    {
        check_dublicate_data::result_state res = fres.get();

        switch (res)
        {
        case check_dublicate_data::ok:
            break;
        case check_dublicate_data::from_himself:
            TASK_LOG(ctx, error) << "task validity check failed: from_himself error";
            break;
        case check_dublicate_data::dublicate:
            TASK_LOG(ctx, error) << "task validity check failed: dublicate";
            break;
        case check_dublicate_data::resolve_fail:
            TASK_LOG(ctx, error) << "task validity check failed: resolve_fail";
            break;
        case check_dublicate_data::list_fail:
            TASK_LOG(ctx, error) << "task validity check failed: list_fail";
            break;
        case check_dublicate_data::access_denied:
            TASK_LOG(ctx, error) << "task validity check failed: access_denied";
            break;
        case check_dublicate_data::blackbox_fail:
            TASK_LOG(ctx, error) << "task validity check failed: blackbox_fail";
            break;
        }

        if (res == check_dublicate_data::from_himself || res == check_dublicate_data::dublicate ||
            res == check_dublicate_data::access_denied)
        {
            prom.set(task_bad);
            return;
        }

        if (res == check_dublicate_data::ok)
        {
            prom.set(task_pq_ok);
            return;
        }

        prom.set(task_temporary_error);
    }
    catch (...)
    {
        prom.set(task_temporary_error);
    }
}

future_connect_result api_impl::check_auth(connect_args args, settings_ptr settings)
{
    if (args.task.use_imap)
    {
        return std::make_shared<imap_auth_op>(args, settings)->run();
    }
    else
    {
        return std::make_shared<pop3_auth_op>(args, settings)->run();
    }
}

} // namespace yrpopper::api
