#include <api/api_impl.h>
#include <api/cache.h>

#include <common/http_loader.h>
#include <oauth/hack/util.h>

#include <yplatform/find.h>

namespace yrpopper::api {

namespace {
struct FuritaLoader : public XmlHttpLoader<HttpGet>
{
public:
    FuritaLoader(
        task_context_ptr context,
        const string& host,
        size_t maxRetries,
        std::string suid,
        std::string mdb)
        : XmlHttpLoader<HttpGet>(context, host, maxRetries), suid(suid), mdb(mdb)
    {
    }

    FutureRpopActionsList getActionList()
    {
        this->load();
        return prom;
    }

protected:
    std::string getRequestData() override
    {
        return "?detailed=1&user=" + suid + "&db=" + mdb;
    }

    void handleXml(boost::property_tree::ptree&& xmlPtree) override
    {
        auto body = xmlPtree.get_child("body");
        for (auto& tag : body)
        {
            if (tag.first == "rule")
            {
                handleRule(tag.second);
            }
        }

        prom.set(res);
    }

    void handleError(const std::string& errorMessage) override
    {
        prom.set_exception(std::runtime_error(errorMessage));
    }

private:
    void handleRule(const boost::property_tree::ptree& rulePtree)
    {
        popid_t popid = 0;
        action_list_t actions;

        for (auto& tag : rulePtree)
        {
            if (tag.first == "condition")
            {
                if ("X-yandex-rpop-id" == tag.second.get<std::string>("<xmlattr>.div", ""))
                {
                    try
                    {
                        popid = boost::lexical_cast<popid_t>(
                            tag.second.get<std::string>("pattern", ""));
                    }
                    catch (...)
                    {
                    }
                }
            }
            else if (tag.first == "action")
            {
                try
                {
                    actions.push_back(boost::lexical_cast<actionid_t>(
                        tag.second.get<std::string>("parameter", "")));
                }
                catch (...)
                {
                }
            }
        }
        if (popid != 0 && !actions.empty())
        {
            res.push_back(std::make_shared<RpopActions>(RpopActions{ popid, actions }));
        }
    }

private:
    PromiseRpopActionsList prom;
    RpopActionsList res;

    std::string suid;
    std::string mdb;
};
}

future_string api_impl::owner(const yplatform::task_context_ptr& ctx, const popid_t& id)
{
    promise_string prom;
    auto fres = dbInterface->getCollectorOwner(ctx, id);
    fres.add_callback(boost::bind(&api_impl::handle_result<std::string>, this, prom, fres));
    return prom;
}

FutureSmtpData api_impl::getSmtpData(
    const yplatform::task_context_ptr& ctx,
    const string& suid,
    const popid_t& id,
    const string& email)
{
    PromiseSmtpData prom;

    auto futureList = this->list(ctx, "", suid, id, false);
    futureList.add_callback([this, ctx, email, id, suid, futureList, prom]() {
        this->handle_smtp_list(ctx, email, id, suid, futureList, prom);
    });

    return prom;
}

FutureStatusData api_impl::hacked_status(const yplatform::task_context_ptr& ctx, popid_t popid)
{
    PromiseStatusData prom;
    auto fres = dbInterface->info(ctx, popid);
    fres.add_callback([this, ctx, popid, fres, prom]() {
        this->handle_hacked_status_collector_info(ctx, popid, fres, prom);
    });

    return prom;
}

void api_impl::handle_hacked_status_collector_info(
    const yplatform::task_context_ptr& ctx,
    popid_t popid,
    future_task_info_list collectorInfo,
    PromiseStatusData prom)
{
    try
    {
        auto collectors = collectorInfo.get();
        if (collectors->empty())
        {
            prom.set_exception(not_found_error());
            return;
        }

        auto&& collector = collectors->front();
        bool isOauth = collector.oauth_refresh_token.size();
        if (isOauth && oauth::hack::OUTLOOK_IMAP_SERVER == collector.server)
        {
            // TODO: avoid copy in handle_result
            auto futureStatus = dbInterface->httpStatus(ctx, popid);
            futureStatus.add_callback(
                boost::bind(&api_impl::handle_result<StatusData>, this, prom, futureStatus));
        }
        else
        {
            auto futureStatus = dbInterface->status(ctx, popid);
            futureStatus.add_callback([this, ctx, popid, futureStatus, prom]() {
                this->handle_status_data(ctx, popid, futureStatus, prom);
            });
        }
    }
    catch (...)
    {
        prom.set_current_exception();
    }
}

FutureStatusData api_impl::status(const yplatform::task_context_ptr& ctx, popid_t popid)
{
    PromiseStatusData prom;

    auto futureStatus = dbInterface->status(ctx, popid);
    futureStatus.add_callback([this, ctx, popid, futureStatus, prom]() {
        this->handle_status_data(ctx, popid, futureStatus, prom);
    });

    return prom;
}

void api_impl::handle_status_data(
    const yplatform::task_context_ptr& ctx,
    popid_t popid,
    FutureStatusData futureStatus,
    PromiseStatusData prom)
{
    auto futureFolders = dbInterface->loadImapFolders(ctx, popid);
    futureFolders.add_callback([this, futureFolders, futureStatus, prom]() {
        this->handle_status_with_folders(futureFolders, futureStatus, prom);
    });
}

void api_impl::handle_status_with_folders(
    future_imap_folders futureFolders,
    FutureStatusData futureStatus,
    PromiseStatusData prom)
{
    try
    {
        auto statuses = futureStatus.get();
        auto folders = futureFolders.get();

        for (auto& folderStatus : statuses)
        {
            for (const auto& folder : *folders)
            {
                if (folder.second->folder_id == folderStatus.folderId)
                {
                    folderStatus.folderName = folder.first;
                    break;
                }
            }
        }

        if (statuses.empty())
        {
            prom.set_exception(collector_not_started_yet());
        }
        else
        {
            prom.set(std::move(statuses));
        }
    }
    catch (...)
    {
        prom.set_current_exception();
    }
}

future_task_info_list api_impl::list_no_cache(
    const yplatform::task_context_ptr& ctx,
    const string& mdb,
    const string& suid,
    const popid_t& id)
{
    promise_task_info_list prom;

    future_task_info_list fRes =
        (id ? dbInterface->info(ctx, suid, id) : dbInterface->list(ctx, suid));
    fRes.add_callback([this, ctx, mdb, suid, fRes, prom]() {
        this->handle_list_no_cache(ctx, mdb, suid, fRes, prom);
    });

    return prom;
}

void api_impl::handle_smtp_list(
    const yplatform::task_context_ptr& ctx,
    const string& email,
    popid_t popid,
    const string& suid,
    future_task_info_list futureList,
    PromiseSmtpData prom)
{
    try
    {
        auto list = futureList.get();

        std::shared_ptr<SmtpData> data;
        for (auto& task : *list)
        {
            if (popid && task.popid != popid) continue;
            if (!popid && !boost::algorithm::iequals(task.email, email)) continue;

            data = std::make_shared<SmtpData>();
            data->login = task.login;
            data->oauth = task.oauth_refresh_token.size();
            data->pass = decrypt_password(
                (data->oauth ? task.oauth_refresh_token : task.password), settings_->dkeys, suid);

            size_t pos = task.email.find('@');
            if (pos != task.email.npos)
            {
                string domain = task.email.substr(pos + 1);

                auto found = settings_->predefined_settings.find(domain);
                if (found != settings_->predefined_settings.end())
                    data->connectData = found->second;
            }

            break;
        }
        if (!data)
        {
            return prom.set_exception(std::runtime_error("no such collector"));
        }

        if (data->oauth)
        {
            auto oauthModule = yplatform::find<oauth::OauthService>("oauth_module");
            auto futureToken =
                oauthModule->getAccessToken(ctx, data->connectData.recieve.server, data->pass);
            futureToken.add_callback([data, prom, futureToken]() mutable {
                if (futureToken.has_exception())
                {
                    prom.set_exception(std::runtime_error(
                        "get access token exception: " + get_exception_reason(futureToken)));
                }
                else
                {
                    data->pass = futureToken.get();
                    prom.set(*data);
                }
            });
        }
        else
        {
            prom.set(*data);
        }
    }
    catch (...)
    {
        prom.set_current_exception();
    }
}

void api_impl::handle_list_no_cache(
    const yplatform::task_context_ptr& ctx,
    const string& mdb,
    const string& suid,
    future_task_info_list fres,
    promise_task_info_list prom)
{
    task_info_list_ptr taskList;
    try
    {
        taskList = fres.get();
    }
    catch (...)
    {
        prom.set_current_exception();
        return;
    }

    if (mdb.empty())
    {
        prom.set(taskList);
        return;
    }

    if (!settings_->useFurita)
    {
        prom.set(taskList);
        return;
    }

    auto loader = boost::make_shared<FuritaLoader>(
        ctx, settings_->furita_host, settings_->furita_retries, suid, mdb);
    auto futureActions = loader->getActionList();
    futureActions.add_callback([this, taskList, futureActions, prom]() {
        this->handle_list_actions(taskList, futureActions, prom);
    });
}

void api_impl::handle_list_actions(
    task_info_list_ptr taskList,
    FutureRpopActionsList futureActions,
    promise_task_info_list prom)
{
    try
    {
        auto actionList = futureActions.get();
        for (auto& task : *taskList)
        {
            auto popid = task.popid;
            auto res = std::find_if(
                actionList.begin(), actionList.end(), [&](const RpopActionsPtr& actions) {
                    return actions->popid == popid;
                });

            if (res != actionList.end())
            {
                task.action = std::move((*res)->actions);
            }
        }

        prom.set(taskList);
    }
    catch (...)
    {
        prom.set_current_exception();
    }
}

future_task_info_list api_impl::list(
    const yplatform::task_context_ptr& ctx,
    const std::string& mdb,
    const std::string& suid,
    const popid_t& id,
    bool skipCache)
{
    promise_task_info_list prom;
    if (!settings_->use_cache || skipCache)
    {
        future_task_info_list fres = list_no_cache(ctx, mdb, suid, id);
        fres.add_callback(
            boost::bind(&api_impl::handle_result<task_info_list_ptr>, this, prom, fres));
    }
    else
    {
        auto cacheService = yplatform::find<ymod_cache::cache>("cache_core");
        ymod_cache::future_segment fres = cacheService->get(ctx, pack(boost::make_tuple(suid, id)));
        fres.add_callback(
            boost::bind(&api_impl::list_cache_get, this, ctx, prom, fres, mdb, suid, id));
    }
    return prom;
}

void api_impl::list_cache_get(
    const yplatform::task_context_ptr& ctx,
    promise_task_info_list prom,
    ymod_cache::future_segment res,
    const std::string& mdb,
    const std::string& suid,
    const popid_t& id)
{
    bool get_failed = false;
    try
    {
        boost::optional<yplatform::zerocopy::segment> s = res.get().value;
        if (s)
        {
            TASK_LOG(ctx, info) << "/api/list cache hit: suid=" << suid << ", popid=" << id;
            task_info_list_ptr task = unpack<task_info_list_ptr>(s.get());
            prom.set(task);
            return;
        }
    }
    catch (std::exception& e)
    {
        TASK_LOG(ctx, error) << "/api/list cache error (get): " << e.what();
        get_failed = true;
    }
    catch (...)
    {
        get_failed = true;
    }

    if (get_failed)
    {
        auto cacheService = yplatform::find<ymod_cache::cache>("cache_core");
        // remove the bad cache value and refetch it from the backend
        auto futureCache = cacheService->remove(ctx, pack(boost::make_tuple(suid, id)));
        auto cacheCallback = boost::bind(
            &api_impl::list_cache_get_remove, this, ctx, prom, futureCache, mdb, suid, id);
        futureCache.add_callback(cacheCallback);
    }
    else
    {
        auto futureList = list_no_cache(ctx, mdb, suid, id);
        auto listCallback =
            boost::bind(&api_impl::list_cache_get_cont, this, ctx, prom, futureList, mdb, suid, id);
        futureList.add_callback(listCallback);
    }
}

void api_impl::list_cache_get_remove(
    const yplatform::task_context_ptr& ctx,
    promise_task_info_list prom,
    ymod_cache::future_result fres,
    const std::string& mdb,
    const std::string& suid,
    const popid_t& id)
{
    try
    {
        fres.get();
    }
    catch (std::exception& e)
    {
        TASK_LOG(ctx, error) << "/api/list cache error (remove): " << e.what();
    }
    catch (...)
    {
    }

    auto futureList = list_no_cache(ctx, mdb, suid, id);
    auto listCallback =
        boost::bind(&api_impl::list_cache_get_cont, this, ctx, prom, futureList, mdb, suid, id);
    futureList.add_callback(listCallback);
}

void api_impl::list_cache_get_cont(
    const yplatform::task_context_ptr& ctx,
    promise_task_info_list prom,
    future_task_info_list res,
    const std::string& /* mdb */,
    const std::string& suid,
    const popid_t& id)
{
    if (res.has_exception())
    {
        TASK_LOG(ctx, error) << "api_impl::list_cache_get_cont exception:"
                             << get_exception_reason(res);
        prom.set_exception(storage_error());
        return;
    }

    task_info_list_ptr tasks = res.get();
    prom.set(tasks);

    auto cacheService = yplatform::find<ymod_cache::cache>("cache_core");
    auto futureResult = cacheService->set(ctx, pack(boost::make_tuple(suid, id)), pack(tasks));
    futureResult.add_callback(
        boost::bind(&api_impl::list_cache_set_tasks, this, ctx, tasks, suid, id, futureResult));
}

void api_impl::list_cache_set_tasks(
    const yplatform::task_context_ptr& ctx,
    task_info_list_ptr tasks,
    const std::string& suid,
    const popid_t& id,
    ymod_cache::future_result fres)
{
    try
    {
        fres.get();
    }
    catch (std::exception& e)
    {
        TASK_LOG(ctx, error) << "/api/list cache error (set): " << e.what();
    }
    catch (...)
    {
    }

    if (id == 0 && !tasks->empty())
    {
        task_info_list::const_iterator it = tasks->begin();
        task_info_list::const_iterator it_next = it;
        ++it_next;

        task_info_list_ptr task(new task_info_list(it, it_next));
        auto cacheService = yplatform::find<ymod_cache::cache>("cache_core");
        ymod_cache::future_result fres =
            cacheService->set(ctx, pack(boost::make_tuple(suid, it->popid)), pack(task));
        auto cacheCallback =
            boost::bind(&api_impl::list_cache_set_task, this, ctx, it_next, tasks, suid, fres);
        fres.add_callback(cacheCallback);
    }
}

void api_impl::list_cache_set_task(
    const yplatform::task_context_ptr& ctx,
    task_info_list::const_iterator it,
    task_info_list_ptr tasks,
    const std::string& suid,
    ymod_cache::future_result fres)
{
    try
    {
        fres.get();
    }
    catch (std::exception& e)
    {
        TASK_LOG(ctx, error) << "/api/list cache error (set): " << e.what();
    }
    catch (...)
    {
    }

    if (it != tasks->end())
    {
        task_info_list::const_iterator it_next = it;
        ++it_next;

        task_info_list_ptr task(new task_info_list(it, it_next));
        auto cacheService = yplatform::find<ymod_cache::cache>("cache_core");
        ymod_cache::future_result fres =
            cacheService->set(ctx, pack(boost::make_tuple(suid, it->popid)), pack(task));
        auto cacheCallback =
            boost::bind(&api_impl::list_cache_set_task, this, ctx, it_next, tasks, suid, fres);
        fres.add_callback(cacheCallback);
    }
}

} // namespace yrpopper::api
