#pragma once

#include <api/api_impl.h>

#include <yplatform/coroutine.h>
#include <yplatform/yield.h>

namespace yrpopper { namespace api {

struct check_rate_control_op : std::enable_shared_from_this<check_rate_control_op>
{
    using yield_context = yplatform::yield_context<check_rate_control_op>;

    promise_void_t res;
    connect_args args;
    rc_options options;

    int tries_count;

    check_rate_control_op(const connect_args& args, const rc_options& options)
        : args(args), options(options)
    {
    }

    future_void_t run()
    {
        yplatform::spawn(yplatform::shared_from(this));
        return res;
    }

    void operator()(yield_context yield_ctx)
    {
        reenter(yield_ctx)
        {
            if (args.rc_email)
            {
                yield probe_rate_control(args.rc_email, args.task.login)
                    .then(yield_ctx.capture(tries_count));
                if (tries_count > options.max_email_count)
                {
                    res.set_exception(rc_email_error());
                    yield break;
                }
            }
            if (args.rc_ip)
            {
                yield probe_rate_control(args.rc_ip, args.ip).then(yield_ctx.capture(tries_count));
                if (tries_count > options.max_ip_count)
                {
                    res.set_exception(rc_ip_error());
                    yield break;
                }
            }
            res.set({});
        }
    }

    void operator()(yield_context::exception_type exception)
    {
        LERR_(args.ctx) << "exception during check_rate_control_op occured: "
                        << get_exception_reason(exception);
        // If rc server down - ignore error
        res.set({});
    }

    future_int_t probe_rate_control(boost::shared_ptr<rc_client> client, const std::string& key)
    {
        promise_int_t res;
        client->probe(
            rc_client::key(key),
            args.ctx->uniq_id(),
            [client, res, ctx = args.ctx](auto ec, auto endpoint) mutable {
                if (ec)
                {
                    LERR_(ctx) << "pop.rc probe error: " << ec.message() << ", host: " << endpoint;
                    res.set_exception(std::runtime_error(ec.message()));
                    return;
                }
                res.set(client->info().counter);
            });
        return res;
    }
};

} // namespace api
} // namespace yrpopper

#include <yplatform/unyield.h>
