#pragma once

#include "check_rate_control_op.h"
#include "update_failed_auth_counter_op.h"
#include "allowed_ip.h"

#include <common/data_source.h>
#include <api/api_impl.h>
#include <api/util_blackbox.h>
#include <api/error.h>

#include <ymod_imapclient/errors.h>
#include <yplatform/coroutine.h>
#include <yplatform/yield.h>

namespace yrpopper { namespace api {

using future_imap_client_t = yplatform::future::future<ymod_imap_client::ImapClientPtr>;

struct imap_auth_op : std::enable_shared_from_this<imap_auth_op>
{
    using yield_context = yplatform::yield_context<imap_auth_op>;

    promise_connect_result res;
    connect_args args;
    settings_ptr settings;

    std::string ip;
    imap_client_ptr_t client;
    ymod_imap_client::ImapCapabilityPtr capabilities;
    std::string server_name;
    std::string access_token;
    std::exception_ptr auth_exception;

    imap_auth_op(const connect_args& args, settings_ptr settings) : args(args), settings(settings)
    {
    }

    future_connect_result run()
    {
        yplatform::spawn(yplatform::shared_from(this));
        return res;
    }

    void operator()(yield_context yield_ctx)
    {
        reenter(yield_ctx)
        {
            yield check_rate_control().then(yield_ctx);
            yield resolve_ip(args.task.server).then(yield_ctx.capture(ip));
            ensure_allowed_ip(ip, settings, args.ctx);
            yield connect(ip).then(yield_ctx.capture(client));
            yield get_capabilities().then(yield_ctx.capture(capabilities));
            if (id_supported(capabilities))
            {
                yield identify_server().then(yield_ctx.capture(server_name));
                if (server_forbidden(server_name))
                {
                    res.set_exception(forbidden_server_error());
                    yield break;
                }
            }
            if (use_oauth())
            {
                yield get_access_token().then(yield_ctx.capture(access_token));
                yield auth_by_oauth(access_token).then(yield_ctx.capture(auth_exception));
            }
            else
            {
                yield auth_by_login().then(yield_ctx.capture(auth_exception));
            }
            yield close_connection().then(yield_ctx);

            if (auth_exception)
            {
                LERR_(args.ctx) << "imap_auth_op fail reason: "
                                << get_exception_reason(auth_exception);
                yield update_failed_auth_counter().then(yield_ctx);
                res.set_exception(auth_exception);
                yield break;
            }
            res.set({ ip });
        }
    }

    void operator()(yield_context::exception_type exception)
    {
        res.set_exception(exception);
    }

    future_void_t check_rate_control()
    {
        return std::make_shared<check_rate_control_op>(args, settings->rc)->run();
    }

    FutureStringResult resolve_ip(const std::string& server)
    {
        auto pop3Service = yplatform::find<ymod_pop_client::call>("pop_client");
        return pop3Service->resolve(args.ctx, server).then([](auto future) {
            return *future.get();
        });
    }

    future_imap_client_t connect(const std::string& ip)
    {
        auto imapService = yplatform::find<ymod_imap_client::call>("imap_client");
        auto client = imapService->clientInstance(args.ctx);
        return client->connect(ip, args.task.port, args.task.use_ssl)
            .then([client, this, self = shared_from_this()](auto future) {
                try
                {
                    future.get();
                }
                catch (const std::exception& e)
                {
                    LERR_(args.ctx)
                        << "Connect error for host " << args.task.server << ": " << e.what();
                    throw connect_error();
                }
                return client;
            });
    }

    ymod_imap_client::FutureCapability get_capabilities()
    {
        return client->capability();
    }

    bool id_supported(ymod_imap_client::ImapCapabilityPtr capabilities)
    {
        return capabilities->capability.id;
    }

    FutureStringResult identify_server()
    {
        return client->id().then([](auto future) {
            auto res = future.get();
            auto it = res->data.find("name");
            if (it == res->data.end()) return ""s;
            return it->second;
        });
    }

    bool server_forbidden(const std::string& name)
    {
        auto begin = settings->forbidden_imap_names.begin();
        auto end = settings->forbidden_imap_names.end();
        return std::find(begin, end, name) != end;
    }

    bool use_oauth()
    {
        return args.task.oauth_refresh_token.size();
    }

    FutureStringResult get_access_token()
    {
        auto oauthModule = yplatform::find<oauth::OauthService>("oauth_module");
        return oauthModule->getAccessToken(
            args.ctx, args.task.server, args.task.oauth_refresh_token);
    }

    future_exception_ptr auth_by_oauth(const std::string& access_token)
    {
        return client->loginOauth(args.task.login, access_token, get_login_type(args.task.server))
            .then([self = shared_from_this(), this](auto future) {
                return process_login_result(future);
            });
    }

    future_exception_ptr auth_by_login()
    {
        return client->login(args.task.login, args.task.password)
            .then([self = shared_from_this(), this](auto future) {
                return process_login_result(future);
            });
        ;
    }

    future_void_t close_connection()
    {
        // Just ignore all errors on logout
        return client->logout().then([](auto /*future*/) { return VoidResult{}; });
    }

    future_void_t update_failed_auth_counter()
    {
        return std::make_shared<update_failed_auth_counter_op>(args, settings->rc)->run();
    }

    ymod_imap_client::ImapClient::OauthLoginType get_login_type(const std::string& server)
    {
        ymod_imap_client::ImapClient::OauthLoginType loginType =
            ymod_imap_client::ImapClient::Multiline;
        for (auto& [name, val] : settings->oauth2Workaround)
        {
            if (name != server) continue;
            loginType = val ? ymod_imap_client::ImapClient::SinglelineLiteral :
                              ymod_imap_client::ImapClient::SinglelineQuoted;
            break;
        }
        return loginType;
    }

    std::exception_ptr process_login_result(ymod_imap_client::FutureImapResult future)
    {
        if (future.has_exception())
        {
            auto fail_reason = get_exception_reason(future);
            return std::make_exception_ptr(login_error(YRPOPPER_API_LOGIN_ERROR_DESC, fail_reason));
        }
        else if (!future.get()->ok)
        {
            return std::make_exception_ptr(login_error());
        }
        return nullptr;
    }
};

} // namespace api
} // namespace yrpopper

#include <yplatform/unyield.h>
