#pragma once

#include "check_rate_control_op.h"
#include "update_failed_auth_counter_op.h"
#include "allowed_ip.h"

#include <common/data_source.h>
#include <api/api_impl.h>
#include <api/util_blackbox.h>
#include <api/error.h>

#include <ymod_imapclient/errors.h>
#include <yplatform/coroutine.h>
#include <yplatform/yield.h>

namespace yrpopper { namespace api {

using future_imap_client_t = yplatform::future::future<ymod_imap_client::ImapClientPtr>;

struct pop3_auth_op : std::enable_shared_from_this<pop3_auth_op>
{
    using yield_context = yplatform::yield_context<pop3_auth_op>;

    promise_connect_result res;
    connect_args args;
    settings_ptr settings;

    std::string ip;
    std::string welcome_message;
    std::exception_ptr auth_exception;

    pop3_auth_op(const connect_args& args, settings_ptr settings) : args(args), settings(settings)
    {
    }

    future_connect_result run()
    {
        yplatform::spawn(yplatform::shared_from(this));
        return res;
    }

    void operator()(yield_context yield_ctx)
    {
        reenter(yield_ctx)
        {
            yield check_rate_control().then(yield_ctx);
            yield resolve_ip(args.task.server).then(yield_ctx.capture(ip));
            ensure_allowed_ip(ip, settings, args.ctx);
            yield connect(ip).then(yield_ctx.capture(welcome_message));
            if (server_forbidden(welcome_message))
            {
                res.set_exception(forbidden_server_error());
                yield break;
            }
            yield auth_by_login().then(yield_ctx.capture(auth_exception));
            yield close_connection().then(yield_ctx);

            if (auth_exception)
            {
                LERR_(args.ctx) << "pop3_auth_op fail reason: "
                                << get_exception_reason(auth_exception);
                yield update_failed_auth_counter().then(yield_ctx);
                res.set_exception(auth_exception);
                yield break;
            }
            res.set({ ip });
        }
    }

    void operator()(yield_context::exception_type exception)
    {
        res.set_exception(exception);
    }

    future_void_t check_rate_control()
    {
        return std::make_shared<check_rate_control_op>(args, settings->rc)->run();
    }

    FutureStringResult resolve_ip(const std::string& server)
    {
        auto pop3Service = yplatform::find<ymod_pop_client::call>("pop_client");
        return pop3Service->resolve(args.ctx, server).then([](auto future) {
            return *future.get();
        });
    }

    FutureStringResult connect(const std::string& ip)
    {
        auto pop3Service = yplatform::find<ymod_pop_client::call>("pop_client");
        return pop3Service->connect(args.ctx, ip, args.task.port, args.task.use_ssl)
            .then([ip = ip, ctx = args.ctx](auto future) {
                try
                {
                    return future.get().welcome_message;
                }
                catch (const std::exception& e)
                {
                    LERR_(ctx) << "Connect error for ip " << ip << ": " << e.what();
                    throw connect_error();
                }
            });
    }

    bool server_forbidden(const std::string& welcome_message)
    {
        for (const auto& m : settings->forbidden_pop3_messages)
        {
            auto pos = welcome_message.find(m);
            if (pos != std::string::npos)
            {
                return true;
            }
        }
        return false;
    }

    future_exception_ptr auth_by_login()
    {
        auto pop3Service = yplatform::find<ymod_pop_client::call>("pop_client");
        return pop3Service->login(args.ctx, args.task.login, args.task.password)
            .then([self = shared_from_this(), this](auto future) {
                if (future.has_exception())
                {
                    auto fail_reason = get_exception_reason(future);
                    return std::make_exception_ptr(
                        login_error(YRPOPPER_API_LOGIN_ERROR_DESC, fail_reason));
                }
                else if (!future.get())
                {
                    return std::make_exception_ptr(login_error());
                }
                return std::exception_ptr();
            });
        ;
    }

    future_void_t close_connection()
    {
        auto pop3Service = yplatform::find<ymod_pop_client::call>("pop_client");
        // Just ignore all errors on logout
        return pop3Service->quit(args.ctx).then([](auto /*future*/) { return VoidResult{}; });
    }

    future_void_t update_failed_auth_counter()
    {
        return std::make_shared<update_failed_auth_counter_op>(args, settings->rc)->run();
    }
};

} // namespace api
} // namespace yrpopper

#include <yplatform/unyield.h>
