#include <api/api_impl.h>
#include <api/util_blackbox.h>

#include <ymod_popclient/errors.h>
#include <ymod_imapclient/errors.h>

#include <yplatform/find.h>

namespace yrpopper::api {

namespace {

inline bool parse_email(const std::string& email, std::string& name, std::string& domain)
{
    if (const char* at = strchr(email.c_str(), '@'))
    {
        name = std::string(email.c_str(), at);
        domain = std::string(at + 1, email.c_str() + email.size());
        return true;
    }
    return false;
}

bool compare_logins(
    const std::string& lh,
    const std::string& lh_server,
    const std::string& rh,
    const std::string& rh_server)
{
    std::string lh_login = lh, lh_domain;
    std::string rh_login = rh, rh_domain;
    if (parse_email(lh, lh_login, lh_domain) && parse_email(rh, rh_login, rh_domain))
    {
        return boost::iequals(lh_login, rh_login) && boost::iequals(lh_domain, rh_domain);
    }

    return boost::iequals(lh_login, rh_login) && boost::iequals(lh_server, rh_server);
}

} // namespace <anonimus>

future_dublicate api_impl::check_dublicate(
    const yplatform::task_context_ptr& ctx,
    const std::string& mdb,
    const std::string& suid,
    popid_t id,
    const std::string& server,
    const std::string& login)
{
    check_dublicate_args args(new check_dublicate_data);
    args->ctx = ctx;
    args->mdb = mdb;
    args->suid = suid;
    args->id = id;
    args->server = server;
    args->login = login;

    auto pop3Service = yplatform::find<ymod_pop_client::call>("pop_client");
    auto futureResolve = pop3Service->resolve(args->ctx, args->server);
    futureResolve.add_callback(boost::bind(&api_impl::handle_resolv_my, this, args, futureResolve));
    return args->prom;
}

void api_impl::handle_resolv_my(
    const check_dublicate_args& args,
    ymod_pop_client::future_string_ptr res)
{
    if (res.has_exception())
    {
        args->prom.set(check_dublicate_data::resolve_fail);
        return;
    }

    args->is_yandex_host = settings_->yandex_nets.contains(*res.get());

    if (args->is_yandex_host &&
        (settings_->allowed_ip.find(*res.get()) == settings_->allowed_ip.end()))
    {
        args->prom.set(check_dublicate_data::access_denied);
        return;
    }

    if (args->is_yandex_host)
    {
        future_bool_t fres = compareUsers(args->ctx, args->suid, args->login, *settings_);
        fres.add_callback(boost::bind(&api_impl::handle_resolv_my_cont, this, args, fres));
    }
    else
    {
        future_task_info_list fres = list(args->ctx, args->mdb, args->suid, 0, false);
        fres.add_callback(boost::bind(&api_impl::handle_list_dub, this, args, fres));
    }
}

void api_impl::handle_resolv_my_cont(const check_dublicate_args& args, future_bool_t fres)
{
    try
    {
        bool identical_emails = fres.get();
        if (identical_emails)
        {
            args->prom.set(check_dublicate_data::from_himself);
            return;
        }

        future_task_info_list fres = list(args->ctx, args->mdb, args->suid, 0, false);
        fres.add_callback(boost::bind(&api_impl::handle_list_dub, this, args, fres));
    }
    catch (std::exception& e)
    {
        args->prom.set_exception(e);
    }
}

void api_impl::handle_list_dub(const check_dublicate_args& args, future_task_info_list res)
{
    if (res.has_exception())
    {
        args->prom.set(check_dublicate_data::list_fail);
        return;
    }

    task_info_list_ptr lst = res.get();

    // TASK_LOG(args->ctx, debug) << "Check duplicate, list recieved:";
    // for (auto& item : *lst) {
    //     TASK_LOG(args->ctx, debug) << "Item: popid=" << item.popid << ", login=" << item.login <<
    //     ", server=" << item.server;
    // }

    if (args->is_yandex_host)
    {
        std::string email = args->login;
        if (email.find('@') == std::string::npos) email += "@yandex.ru";

        auto futureSuid = getSuid(args->ctx, email, *settings_);
        futureSuid.add_callback(
            boost::bind(&api_impl::handle_list_dub_yandex, this, args, futureSuid, lst));
        return;
    }
    else
    {
        for (auto& i : *lst)
        {
            if (i.popid == args->id) continue;

            if (compare_logins(args->login, args->server, i.login, i.server))
            {
                TASK_LOG(args->ctx, debug) << "Found duplicate: popid=" << i.popid
                                           << ", login=" << i.login << ", server=" << i.server;
                args->prom.set(check_dublicate_data::dublicate);
                return;
            }
        }
        args->prom.set(check_dublicate_data::ok);
    }
}

void api_impl::handle_list_dub_yandex(
    const check_dublicate_args& args,
    FutureStringResult fres,
    task_info_list_ptr lst)
{
    if (fres.has_exception())
    {
        args->prom.set(check_dublicate_data::blackbox_fail);
        return;
    }

    std::string suid = fres.get();
    try
    { // symtomatic solution for empty shared_ptr
        args->resolve_size = args->resolve_count = 0;
        for (auto& i : *lst)
        {
            if (i.popid == args->id) continue;
            ++args->resolve_size;
        }

        if (args->resolve_count >= args->resolve_size)
        {
            args->prom.set(check_dublicate_data::ok);
            return;
        }

        for (auto& i : *lst)
        {
            if (i.popid == args->id) continue;

            auto pop3Service = yplatform::find<ymod_pop_client::call>("pop_client");
            auto futureResolve = pop3Service->resolve(args->ctx, i.server);
            auto resolveCallback =
                boost::bind(&api_impl::handle_resolv_dub, this, args, futureResolve, suid, i);
            futureResolve.add_callback(resolveCallback);
        }
    }
    catch (const std::exception& e)
    {
        TASK_LOG(args->ctx, error) << "api_impl::handle_list_dub_yandex exception: " << e.what();
        args->prom.set(check_dublicate_data::list_fail);
    }
}

void api_impl::handle_resolv_dub(
    const check_dublicate_args& args,
    ymod_pop_client::future_string_ptr res,
    const std::string& suid,
    task_info i)
{
    if (!res.has_exception())
    {
        bool is_yandex_host = settings_->yandex_nets.contains(*res.get());
        if (is_yandex_host)
        {
            future_bool_t fres = compareUsers(args->ctx, suid, i.login, *settings_);
            fres.add_callback(boost::bind(&api_impl::handle_resolv_dub_cont, this, args, fres, i));
            return;
        }
    }

    ++args->resolve_count;
    if (args->resolve_count < args->resolve_size) return;
    args->prom.set(check_dublicate_data::ok);
}

void api_impl::handle_resolv_dub_cont(
    const check_dublicate_args& args,
    future_bool_t fres,
    task_info i)
{
    try
    {
        bool identical_emails = fres.get();
        if (identical_emails)
        {
            TASK_LOG(args->ctx, debug) << "Found duplicate: popid=" << i.popid
                                       << ", login=" << i.login << ", server=" << i.server;
            args->prom.set(check_dublicate_data::dublicate);
            return;
        }

        ++args->resolve_count;
        if (args->resolve_count < args->resolve_size) return;
        args->prom.set(check_dublicate_data::ok);
    }
    catch (std::exception& e)
    {
        args->prom.set_exception(e);
    }
}

} // namespace yrpopper::api
