#pragma once

#include <iterator>
#include <boost/asio.hpp>
#include <boost/format.hpp>
#include <deque>
#include <yadns/basic_dns_resolver.h>
#include <yadns/ares_resolver_service.h>

typedef basic_dns_resolver<ares_resolver_service> resolver_t;

// Host resolver intented to check hosts in config options
/*
 * Preconditions:
 * 1. Endpoint(std::string(), int()) should be a valid expression convertible to OutIter::value_type
 * 2. InIter models input iterator
 * 3. OutIter models output iterator (may be singular)
 * 4. OutIter::value_type should be default constructible
 */
template <typename Endpoint, typename InIter, typename OutIter>
class host_sequence_resolver
{
    boost::asio::io_service ios_;
    resolver_t r_;

    typedef std::deque<Endpoint> deque_t;
    class resolver
    {
    public:
        InIter src_;
        typename deque_t::iterator dst_;
        unsigned short port_;

        typedef void result_type;

        resolver(InIter src, typename deque_t::iterator dst, int port)
            : src_(src), dst_(dst), port_(static_cast<unsigned short>(port))
        {
        }

        void operator()(const boost::system::error_code& ec, resolver_t::iterator_aaaa it)
        {
            auto notFoundError =
                (ec == boost::asio::error::host_not_found_try_again ||
                 ec == boost::asio::error::host_not_found);
            if (ec && !notFoundError)
            {
                throw std::runtime_error(
                    str(boost::format("failed to resolve %1%") % ec.message()));
            }

            if (it != resolver_t::iterator_aaaa())
                *dst_++ = Endpoint(boost::asio::ip::address::from_string(*it), port_);
        }
    };

public:
    host_sequence_resolver() : ios_(), r_(ios_)
    {
    }

    void operator()(InIter ibeg, InIter iend, OutIter dst, int port)
    {
        deque_t d;
        for (; ibeg != iend; ++ibeg)
        {
            d.push_back(typename deque_t::value_type());
            typename deque_t::iterator dst = --d.end();
            try
            {
                // See if resolving is really needed
                Endpoint e(
                    boost::asio::ip::address::from_string(*ibeg),
                    static_cast<unsigned short>(port));
                *dst++ = e;
                continue;
            }
            catch (...)
            {
            }

            r_.async_resolve_aaaa(*ibeg, resolver(ibeg, dst, port));
        }

        ios_.run();
        std::copy(d.begin(), d.end(), dst);
    }
};

template <typename Endpoint, typename In, typename Out>
void resolve_host_sequence(In ibeg, In iend, Out out, int port)
{
    host_sequence_resolver<Endpoint, In, Out> r;
    r(ibeg, iend, out, port);
}
