#include <boost/functional/hash.hpp>
#include <boost/thread.hpp>
#include <boost/lexical_cast.hpp>
#include <unistd.h>
#include "rc.h"

// --
class basic_rc_category_t : public boost::system::error_category
{
public:
    const char* name() const noexcept override
    {
        return "rpop.rc";
    }

    std::string message(int value) const override
    {
        if (value == rc_client::logical_error)
            return "Logical error (e.g. mark() called before probe())";
        return "rpop.rc error";
    }
};

const boost::system::error_category& get_rc_category()
{
    static basic_rc_category_t instance;
    return instance;
}

namespace boost {
template <class Range>
std::size_t hash_value(const Range& r)
{
    return boost::hash_range(r.begin(), r.end());
}
}

std::size_t hash_value(const rc_client::key& i)
{
    std::size_t seed = 0;
    for (char v : i.email)
        boost::hash_combine(seed, tolower(v));
    return seed;
}

// --
class rc_client::request : public basic_rc_request<request_handler_t>
{
public:
    typedef basic_rc_request<request_handler_t> base;

    handler_t handler;
    int retries_left;

    request(
        boost::asio::io_service& ios,
        boost::asio::ip::udp::endpoint host,
        request_handler_t h,
        handler_t hh,
        int retries)
        : base(ios, host, h), handler(hh), retries_left(retries)
    {
    }
};

rc_client::rc_client(boost::asio::io_service& ios, const rc_options& opt) : opt_(opt), cl_(ios)
{
}

void rc_client::probe(const key& k, const std::string& comment, handler_t handler)
{
    request_handler_t h(boost::bind(&rc_client::handle_probe, shared_from_this(), _1, _2));
    std::size_t keyhash = hash_value(k);
    boost::asio::ip::udp::endpoint host(*(opt_.hosts.begin() + (keyhash % opt_.hosts.size())));

    boost::shared_ptr<request> req(
        new request(cl_.io_service(), host, h, handler, opt_.udp_retries));

    i_.keyhash = keyhash;
    i_.host = host;
    i_.valid = false;
    i_.counter = 0;

    // Encode request_pb
    request_pb& q = req->q_pb;
    q.set_id(basic_rc_request_id_gen(keyhash));
    q.set_command(request_pb::GET);
    q.set_key_namespace(opt_.ns.data());
    q.set_key(boost::lexical_cast<std::string>(keyhash).data());
    q.set_comment(comment.data());

    cl_.start(req, boost::posix_time::seconds(opt_.udp_timeout));
}

void rc_client::mark(const std::string& comment, handler_t handler)
{
    request_handler_t h(boost::bind(&rc_client::handle_mark, shared_from_this(), _1, _2));
    std::size_t keyhash = i_.keyhash;
    const boost::asio::ip::udp::endpoint& host = i_.host;

    if (!i_.valid) return handler(make_error_code(logical_error), host);

    boost::shared_ptr<request> req(
        new request(cl_.io_service(), host, h, handler, opt_.udp_retries));

    request_pb& q = req->q_pb;

    // Encode request_pb
    q.set_id(basic_rc_request_id_gen(keyhash));
    q.set_command(request_pb::ADD);
    q.set_key_namespace(opt_.ns.data());
    q.set_key(boost::lexical_cast<std::string>(keyhash).data());
    q.set_ttl(opt_.record_lifetime);
    q.set_comment(comment.data());
    q.add_param(1);

    cl_.start(req, boost::posix_time::seconds(opt_.udp_timeout));
}

void rc_client::stop()
{
    cl_.stop();
}

void rc_client::handle_probe(const boost::system::error_code& ec, boost::shared_ptr<request> req)
{
    const reply_pb& a = req->reply();

    if (!ec)
    {
        i_.valid = true;
        i_.age = a.age();
        i_.counter = a.result_size() == 0 ? 0 : a.result().Get(0);
    }
    else if (ec != boost::asio::error::operation_aborted && --req->retries_left > 0)
    {
        request_handler_t h(boost::bind(&rc_client::handle_probe, shared_from_this(), _1, _2));
        boost::shared_ptr<request> nreq(
            new request(cl_.io_service(), i_.host, h, req->handler, req->retries_left));
        nreq->q_pb = req->q_pb;
        return cl_.start(nreq, boost::posix_time::seconds(opt_.udp_timeout));
    }

    return req->handler(ec, req->host());
}

void rc_client::handle_mark(const boost::system::error_code& ec, boost::shared_ptr<request> req)
{
    return req->handler(ec, req->host());
}
