#pragma once

#include <boost/asio.hpp>
#include <boost/shared_ptr.hpp>
#include <boost/enable_shared_from_this.hpp>
#include <boost/range.hpp>
#include "rc_options.h"
#include "basic_rc_client.h"
#include <functional>

class rc_client : public boost::enable_shared_from_this<rc_client>
{
public:
    typedef std::vector<boost::asio::ip::udp::endpoint> hostlist;

    enum errors
    {
        logical_error
    };

    struct key
    {
        std::string email;
        key(const std::string& dest_email) : email(dest_email)
        {
        }
    };

    struct info_t // result of rc probing
    {
        std::size_t keyhash;
        boost::asio::ip::udp::endpoint host;
        rc_options opt;
        int counter;
        long long int age; // age of the key
        bool valid;        // wether contents of this struct make any sense (depends on the probe
                           // completion status)

        info_t() : keyhash(0), host(), opt(), counter(0), age(0), valid(false)
        {
        }
    };

    rc_client(boost::asio::io_service& ios, const rc_options& opt);

    typedef std::function<void(const boost::system::error_code&, hostlist::value_type)> handler_t;

    // Start asynchronous rc check; issues a 'get' request
    void probe(const key& i, const std::string& comment, handler_t handler);

    // Get the result of the last probe() compeletion
    const info_t& info() const
    {
        return i_;
    }

    // Issues an asynchronous 'add' request using the result of the last probe() completion
    void mark(const std::string& comment, handler_t handler);

    // Stop the last request
    void stop();

private:
    class request;
    typedef std::function<
        void(const boost::system::error_code&, boost::shared_ptr<rc_client::request>)>
        request_handler_t;

    void handle_probe(const boost::system::error_code& ec, boost::shared_ptr<request> req);
    void handle_mark(const boost::system::error_code& ec, boost::shared_ptr<request> req);

    const rc_options& opt_;
    basic_rc_client<request> cl_;
    info_t i_;
};

const boost::system::error_category& get_rc_category();

static const boost::system::error_category& rc_category = get_rc_category();

inline boost::system::error_code make_error_code(rc_client::errors e)
{
    return boost::system::error_code(static_cast<int>(e), rc_category);
}

namespace boost { namespace system {
template <>
struct is_error_code_enum<rc_client::errors>
{
    static const bool value = true;
};
}}
