#pragma once

#include <vector>
#include <list>
#include <algorithm>
#include <boost/program_options.hpp>
#include <boost/asio.hpp>
#include "host_seq_resolver.h"

struct rc_options
{
    std::string ns;
    ::time_t udp_timeout;
    int udp_retries;
    int udp_port;
    std::vector<boost::asio::ip::udp::endpoint> hosts;
    ::time_t record_lifetime;
    int max_email_count;
    int max_ip_count;
};

struct rc_options_parser
{
    rc_options& p;
    rc_options_parser(rc_options& opt) : p(opt)
    {
    }

    void parse_from_file(const std::string& filename)
    {
        namespace bpo = boost::program_options;
        bpo::options_description descr("ec options");
        bpo::variables_map vm;
        std::vector<std::string> hosts;
        descr.add_options()("ns", bpo::value<std::string>(&p.ns)->default_value("rp"), "ns")(
            "record_lifetime",
            bpo::value<time_t>(&p.record_lifetime)->default_value(604800),
            "record lifetime")(
            "max_email_count",
            bpo::value<int>(&p.max_email_count)->default_value(4),
            "max email count")(
            "max_ip_count", bpo::value<int>(&p.max_ip_count)->default_value(9), "max ip count")(
            "udp_timeout", bpo::value<time_t>(&p.udp_timeout)->default_value(1), "udp timeout")(
            "udp_retries", bpo::value<int>(&p.udp_retries)->default_value(3), "udp retries")(
            "udp_port", bpo::value<int>(&p.udp_port)->default_value(8890), "udp port")(
            "host", bpo::value<std::vector<std::string>>(&hosts)->required(), "rcsrv hosts");
        bpo::store(bpo::parse_config_file<char>(filename.c_str(), descr, false), vm);
        bpo::notify(vm);

        resolve_host_sequence<boost::asio::ip::udp::endpoint>(
            hosts.begin(), hosts.end(), std::back_inserter(p.hosts), p.udp_port);

        if (p.hosts.empty())
        {
            throw std::runtime_error("failed to resolve rc hosts");
        }

        std::sort(p.hosts.begin(), p.hosts.end());
    }
};
