#include <api/util_blackbox.h>

#include <api/api_impl.h>
#include <ymod_blackbox/auth.h>

typedef yplatform::future::future<ymod_blackbox::response> FutureBbResponse;

namespace yrpopper::api {

typedef yplatform::future::promise<SuidMdbPtr> PromiseSuidMdb;
typedef yplatform::future::promise<std::string> PromiseStringResult;

void handleSuidMdb(FutureBbResponse fres, PromiseSuidMdb promise)
{
    try
    {
        ymod_blackbox::response res = fres.get();

        // NO 404 response on bad uid, just empty fields
        if (res.success && (res.storage.empty() || res.suid.empty()))
        {
            res.success = false;
            res.err_verbose = "bad uid";
        }

        if (!res.success)
        {
            promise.set_exception(std::runtime_error(string("blackbox error: ") + res.err_verbose));
            return;
        }

        auto result = std::make_shared<SuidMdb>();
        result->mdb = res.storage;
        result->suid = res.suid;

        promise.set(result);
    }
    catch (...)
    {
        promise.set_current_exception();
    }
}

FutureSuidMdb getSuidMdb(
    yplatform::task_context_ptr ctx,
    std::string uid,
    const ApiSettings& settings)
{
    PromiseSuidMdb res;

    ymod_blackbox::request bbReq(ctx, uid, settings.sid);
    bbReq.no_password = true;
    bbReq.address = settings.my_ip_addr;

    FutureBbResponse futureBlackbox =
        yplatform::find<ymod_blackbox::auth>("auth")->authenticate(bbReq);
    futureBlackbox.add_callback([=]() { handleSuidMdb(futureBlackbox, res); });

    return res;
}

void handlePddUser(FutureBbResponse fres, PromiseBoolResult promise)
{
    try
    {
        ymod_blackbox::response res = fres.get();
        if (!res.success)
        {
            promise.set_exception(std::runtime_error(string("blackbox error: ") + res.err_verbose));
            return;
        }

        promise.set(!res.uid.empty());
    }
    catch (...)
    {
        promise.set_current_exception();
    }
}

FutureBoolResult isPddUser(
    yplatform::task_context_ptr /* ctx */,
    std::string email,
    const ApiSettings& settings)
{
    PromiseBoolResult res;

    ymod_blackbox::request bbReq;
    bbReq.username = email;
    bbReq.service = "smtp";
    bbReq.no_password = true;
    bbReq.sid = settings.sid;
    bbReq.address = settings.my_ip_addr;

    auto futureBlackbox = yplatform::find<ymod_blackbox::auth>("auth")->authenticate(bbReq);
    futureBlackbox.add_callback([=]() { handlePddUser(futureBlackbox, res); });

    return res;
}

void handleCompareUsers(FutureBbResponse fres, std::string suid, PromiseBoolResult promise)
{
    try
    {
        ymod_blackbox::response res = fres.get();
        if (!res.success)
        {
            promise.set_exception(std::runtime_error(string("blackbox error: ") + res.err_verbose));
            return;
        }

        promise.set(res.success && res.suid == suid);
    }
    catch (std::exception& e)
    {
        promise.set_exception(e);
    }
}

FutureBoolResult compareUsers(
    yplatform::task_context_ptr /* ctx */,
    std::string suid,
    std::string email,
    const ApiSettings& settings)
{
    PromiseBoolResult res;

    ymod_blackbox::request bbReq;
    bbReq.username = email;
    bbReq.no_password = true;
    bbReq.sid = settings.sid;
    bbReq.address = settings.my_ip_addr;

    auto futureBlackbox = yplatform::find<ymod_blackbox::auth>("auth")->authenticate(bbReq);
    futureBlackbox.add_callback([=]() { handleCompareUsers(futureBlackbox, suid, res); });

    return res;
}

void handleSuid(FutureBbResponse fres, PromiseStringResult promise)
{
    try
    {
        ymod_blackbox::response res = fres.get();

        if (!res.success)
        {
            promise.set_exception(std::runtime_error(string("blackbox error: ") + res.err_verbose));
            return;
        }

        promise.set(res.suid);
    }
    catch (...)
    {
        promise.set_current_exception();
    }
}

FutureStringResult getSuid(
    yplatform::task_context_ptr /* ctx */,
    std::string email,
    const ApiSettings& settings)
{
    PromiseStringResult res;

    ymod_blackbox::request bbReq;
    bbReq.username = email;
    bbReq.no_password = true;
    bbReq.sid = settings.sid;
    bbReq.address = settings.my_ip_addr;

    auto futureBlackbox = yplatform::find<ymod_blackbox::auth>("auth")->authenticate(bbReq);
    futureBlackbox.add_callback([=]() { handleSuid(futureBlackbox, res); });

    return res;
}

} // namespace yrpopper::api
