#include <collector_ng/collector.h>
#include <collector_ng/collector_error.h>
#include <common/idna.h>
#include <common/karma.h>

#include <yplatform/log.h>
#include <collector_ng/email_validator.h>

namespace yrpopper::collector {

FutureStepResultPtr CollectorImpl::step()
{
    if (inProgress)
    {
        return currentRes;
    }

    inProgress = true;
    currentRes = PromiseStepResultPtr();
    stepResult = std::make_shared<StepResult>();
    stepResult->validated = context->task->validated;
    stepResult->badRetries = context->task->bad_retries;

    if (!inited)
    {
        init();
    }
    else if (
        context->task->bb_info.getLastUpdated() <
        std::time(0) - settings->blackboxExpirationTimeout)
    {
        updateBlackbox();
    }
    else
    {
        collect();
    }

    return currentRes;
}

void CollectorImpl::finishStep(std::exception_ptr e)
{
    if (e)
    {
        try
        {
            std::rethrow_exception(e);
        }
        catch (const NoSessionError& e)
        {
            stepResult->error = { code::no_session_error, e.what() };
        }
        catch (const BlackboxError& e)
        {
            stepResult->error = { code::blackbox_error, e.what() };
        }
        catch (const KarmaError& e)
        {
            stepResult->error = { code::bad_karma, e.what() };
        }
        catch (const ResolveError& e)
        {
            stepResult->badRetries++;
            stepResult->error = { code::resolve_error, e.what() };
        }
        catch (const ConnectError& e)
        {
            stepResult->badRetries++;
            stepResult->error = { code::connect_error, e.what() };
        }
        catch (const SslError& e)
        {
            stepResult->badRetries++;
            stepResult->error = { code::ssl_error, e.what() };
        }
        catch (const AuthError& e)
        {
            stepResult->badRetries++;
            stepResult->error = { code::login_error, e.what() };
        }
        catch (const DbError& e)
        {
            stepResult->error = { code::db_error, e.what() };
        }
        catch (const TransportError& e)
        {
            stepResult->error = { code::transport_error, e.what() };
        }
        catch (const TimeoutError& e)
        {
            stepResult->error = { code::timeout_error, e.what() };
        }
        catch (const std::exception& e)
        {
            stepResult->error = { code::internal_error, e.what() };
        }
    }

    if (context->task->validated || !inited)
    {
        fullfillPromise();
        return;
    }

    auto validator = boost::make_shared<EmailValidator>(
        context, settings->validatorHost, settings->validatorRetries);
    auto futureValidated = validator->validate();
    auto self = shared_from_this();
    futureValidated.add_callback([this, self, validator, futureValidated]() {
        stepResult->validated = futureValidated.get();
        fullfillPromise();
    });
}

void CollectorImpl::fullfillPromise()
{
    currentRes.set(stepResult);
    session->stop();
    inited = false;
    inProgress = false;
}

void CollectorImpl::init()
{
    TASK_LOG(context, info) << "Initializing collector session";

    if (!session)
    {
        TASK_LOG(context, error) << "Collector has no CollectorSession on start";
        return finishStep(std::make_exception_ptr(NoSessionError()));
    }

    session->init(std::bind(&CollectorImpl::handleInit, shared_from_this(), std::placeholders::_1));
}

void CollectorImpl::collect()
{
    TASK_LOG(context, info) << "Collecting mail";
    session->collect(
        std::bind(&CollectorImpl::handleCollect, shared_from_this(), std::placeholders::_1));
}

void CollectorImpl::updateBlackbox()
{
    TASK_LOG(context, info) << "Updating blackbox info";

    // TODO real ip addr
    ymod_blackbox::request req(context, context->task->suid, 2, "127.0.0.1");
    req.load_language = true;
    req.no_password = true;

    auto authModule = yplatform::find<ymod_blackbox::auth>("auth");
    auto bbInfoRes = authModule->authenticate(req);
    bbInfoRes.add_callback(
        std::bind(&CollectorImpl::handleBlackbox, shared_from_this(), bbInfoRes));
}

void CollectorImpl::handleInit(std::exception_ptr e)
{
    if (e)
    {
        try
        {
            std::rethrow_exception(e);
        }
        catch (const std::exception& e)
        {
            TASK_LOG(context, error)
                << "Initializing collector session failed, exception: '" << e.what() << "'";
        }

        return finishStep(e);
    }

    TASK_LOG(context, info) << "Initializing collector session finished successfully";
    inited = true;
    updateBlackbox();
}

void CollectorImpl::handleBlackbox(FutureAuthResponse bbInfoRes)
{
    if (bbInfoRes.has_exception())
    {
        TASK_LOG(context, error) << "ymod_blackbox exception: '" << get_exception_reason(bbInfoRes)
                                 << "'";
        TASK_LOG(context, info) << "Updating blackbox info failed";
        return finishStep(std::make_exception_ptr(BlackboxError()));
    }

    auto resp = bbInfoRes.get();

    if (!resp.success)
    {
        TASK_LOG(context, error) << "blackbox error: '" << resp.err_str << "', "
                                 << resp.err_verbose;
        TASK_LOG(context, info) << "Updating blackbox info failed";
        stepResult->badRetries++;
        return finishStep(std::make_exception_ptr(BlackboxError()));
    }

    if (resp.storage.empty())
    {
        TASK_LOG(context, error) << "blackbox error: 'nonexistent user'";
        TASK_LOG(context, info) << "Updating blackbox info failed";
        stepResult->badRetries++;
        return finishStep(std::make_exception_ptr(BlackboxError()));
    }

    if (!checkKarma(resp.karma, resp.karma_status))
    {
        TASK_LOG(context, error) << "blackbox error: 'bad karma'";
        TASK_LOG(context, info) << "Updating blackbox info failed";

        stepResult->badRetries++;
        return finishStep(std::make_exception_ptr(KarmaError()));
    }

    blackbox_task_info bbInfo;
    bbInfo.update(resp.uid, resp.storage, idna_encode_email(resp.email), resp.language);

    session->updateBlackboxInfo(bbInfo);
    TASK_LOG(context, info) << "Updating blackbox info finished successfully";
    collect();
}

void CollectorImpl::handleCollect(std::exception_ptr e)
{
    if (e)
    {
        try
        {
            std::rethrow_exception(e);
        }
        catch (const std::exception& e)
        {
            TASK_LOG(context, error) << "Collecting mail failed, exception: '" << e.what() << "'";
        }
    }
    else
    {
        stepResult->badRetries = 0;
        TASK_LOG(context, info) << "Collecting mail finished successfully";
    }

    finishStep(e);
}

void CollectorImpl::stop()
{
    if (!session)
    {
        // Can happen for POP3 collectors
        return;
    }
    session->stop();
}

rpop_context_ptr CollectorImpl::getContext()
{
    return context;
}

void CollectorImpl::updateContext(rpop_context_ptr context)
{
    if (inProgress)
    {
        // cannot update right now, should never happened
        // if happend - nothing fatal, but log required after NEW scheduler implemented
        return;
    }

    if (!session)
    {
        // Can happen for POP3 collectors
        return;
    }

    this->context = context;
    session->updateContext(context);
}

} // namespace yrpopper::collector
