#pragma once

#include <string>
#include <memory>

#include <yplatform/ptree.h>

namespace yrpopper { namespace collector {

struct OauthLoginInfo
{
    std::string serverName;
    bool literal;
};

struct HttpCollectorSettings
{
    uint32_t maxMessagesPerFolder;
    uint32_t fetchChunkSize;
    uint32_t maxBadRetriesPerMessage;
    uint32_t threads;
    std::string ms365BaseUrl;
};

struct CollectorSettings
{
    void update(const yplatform::ptree& xml)
    {
        blackboxExpirationTimeout = xml.get("blackbox_expiration_timeout", 86400);
        maxMessagesPerFolder = xml.get("max_messages_per_folder", 500);
        fetchChunkSize = xml.get("fetch_chunk_size", 500);
        validatorHost = xml.get("validator_host", "http://validator.yandex.net/validate.xml");
        validatorRetries = xml.get("validator_retries", 3);
        httpCollectorsEnabled = xml.get("http_collectors_enabled", false);
        if (httpCollectorsEnabled)
        {
            auto& http_collectors = xml.get_child("http_collectors");
            httpCollector.maxBadRetriesPerMessage =
                http_collectors.get("max_bad_retries_per_message", 3);
            httpCollector.threads = http_collectors.get("threads", 16);
            if (httpCollectorsEnabled)
                httpCollector.ms365BaseUrl = http_collectors.get<std::string>("ms365_base_url");
            httpCollector.maxMessagesPerFolder =
                http_collectors.get("max_messages_per_folder", maxMessagesPerFolder);
            httpCollector.fetchChunkSize = http_collectors.get("fetch_chunk_size", fetchChunkSize);
        }

        auto folders = xml.get_child("folders");
        for (auto& folder : folders)
        {
            if (folder.first == "folder")
            {
                std::string src = folder.second.get<std::string>("<xmlattr>.src");
                std::string dst = folder.second.get<std::string>("<xmlattr>.dst");

                folderNamesMap.insert(std::make_pair(src, dst));
            }
            else
            {
                throw std::runtime_error(
                    "Wrong config format, not <folder> in <folders>, found: " + folder.first);
            }
        }

        auto servers = xml.get_child("oauth_singleline");
        for (auto& server : servers)
        {
            auto name = server.second.get<std::string>("<xmlattr>.server");
            auto literal = server.second.get("<xmlattr>.literal", "");
            oauthLoginTypes.push_back(OauthLoginInfo{ name, boost::lexical_cast<bool>(literal) });
        }
    }

    uint32_t blackboxExpirationTimeout;
    uint32_t maxMessagesPerFolder;
    uint32_t fetchChunkSize;
    bool httpCollectorsEnabled;
    HttpCollectorSettings httpCollector;

    std::string validatorHost;
    uint32_t validatorRetries;

    std::map<std::string, std::string> folderNamesMap;
    std::vector<OauthLoginInfo> oauthLoginTypes;
};

typedef std::shared_ptr<CollectorSettings> CollectorSettingsPtr;

} // namespace collector
} // namespace yrpopper
