#include <collector_ng/http/ms365_client.h>
#include <collector_ng/collector_error.h>
#include <ymod_httpclient/errors.h>
#include <ymod_httpclient/url_encode.h>
#include <yplatform/find.h>

#include <boost/format.hpp>

#include <algorithm>

namespace yrpopper::collector {

MS365Client::MS365Client(
    rpop_context_ptr ctx,
    const std::string& accessToken,
    const std::string& baseUrl,
    std::uint32_t maxFetchSize)
    : collectorCtx(ctx), accessToken(accessToken), baseUrl(baseUrl), maxFetchSize(maxFetchSize)
{
}

MS365FolderList MS365Client::fetchFolders()
{
    auto root = rootFolder();
    MS365FolderList folders;
    fetchChildFolders(root, folders);
    return folders;
}

MS365Folder MS365Client::rootFolder()
{
    auto url = makeRootFolderUrl();
    auto response = request(url);

    if (response.status != 200)
    {
        throw std::runtime_error(
            "failed to fetch root folder: status=" + std::to_string(response.status) +
            " reason=" + response.reason);
    }

    json_value folder;
    if (auto error = folder.parse(response.body))
    {
        throw std::runtime_error("failed to parse http response body");
    }
    return MS365Folder::fromJson(folder);
}

void MS365Client::fetchChildFolders(const MS365Folder& parent, MS365FolderList& output)
{
    auto url = makeFetchChildFoldersUrl(parent.id);
    auto folders = requestMultiplePages<MS365Folder>(url);

    for (auto&& folder : folders)
    {
        if (!parent.path.empty())
        {
            folder.path = parent.path + MS365_FOLDER_DELIM;
        }
        folder.path += folder.name;
        output.push_back(folder);
        if (folder.childFolderCount > 0)
        {
            fetchChildFolders(folder, output);
        }
    }
}

std::vector<MS365Message> MS365Client::fetchMessages(
    const std::string& fid,
    std::time_t prevModified,
    std::time_t prevReceived,
    std::size_t count)
{
    auto url = makeFetchMessagesUrl(fid, prevModified, prevReceived);
    auto messages = requestMultiplePages<MS365Message>(url, count);
    std::sort(messages.begin(), messages.end());
    if (messages.size() > count)
    {
        messages.resize(count);
    }
    return messages;
}

std::string MS365Client::downloadMessage(const std::string& mid)
{
    auto url = makeDownloadMessageUrl(mid);
    std::string headers = "Prefer: text\r\n";

    auto response = request(url, headers);

    if (response.status != 200)
    {
        throw std::runtime_error(
            "failed to download message: id= " + mid +
            " status=" + std::to_string(response.status) + " reason=" + response.reason);
    }
    return response.body;
}

yhttp::response MS365Client::request(const std::string& url, std::string headers)
{
    auto client = yplatform::find<yhttp::call>("ms365_http_client");
    headers += "Authorization: Bearer " + accessToken + "\r\n";
    return client->run(collectorCtx, yhttp::request::GET(url, std::move(headers)));
}

std::string MS365Client::makeRootFolderUrl() const
{
    std::ostringstream os;
    os << baseUrl
       << boost::format("/users/%1%/mailFolders/msgfolderroot") % collectorCtx->task->email;
    return os.str();
}

std::string MS365Client::makeFetchChildFoldersUrl(const std::string& fid) const
{
    std::ostringstream os;
    os << baseUrl
       << boost::format("/users/%1%/mailFolders/%2%/childFolders") % collectorCtx->task->email %
            fid;
    return os.str();
}

std::string MS365Client::makeFetchMessagesUrl(
    const std::string& fid,
    std::time_t prevModified,
    std::time_t prevReceived) const
{
    std::string selectFields = "receivedDateTime,lastModifiedDateTime,internetMessageHeaders,"
                               "subject,internetMessageId,parentFolderId,isRead";
    std::ostringstream os;
    os << baseUrl
       << boost::format("/users/%1%/mailFolders/%2%/messages") % collectorCtx->task->email % fid
       << yhttp::url_encode({ { "$orderby", "lastModifiedDateTime,receivedDateTime" },
                              { "$filter", makeMessageFilter(prevModified, prevReceived) },
                              { "$select", selectFields },
                              { "$top", std::to_string(maxFetchSize) } });

    return os.str();
}

std::string MS365Client::makeDownloadMessageUrl(const std::string& mid) const
{
    std::ostringstream os;
    os << baseUrl
       << boost::format("/users/%1%/messages/%2%/$value") % collectorCtx->task->email % mid;
    return os.str();
}

std::string MS365Client::makeMessageFilter(
    std::time_t modifiedDateTime,
    std::time_t receivedDateTime) const
{
    auto modifiedStr = datetime_to_string(modifiedDateTime);
    auto nextByReceivedStr = datetime_to_string(receivedDateTime + 1);
    auto nextByModifiedStr = datetime_to_string(modifiedDateTime + 1);
    std::ostringstream os;
    os << boost::format("(lastModifiedDateTime ge %1% and lastModifiedDateTime lt %3% and "
                        "receivedDateTime ge %2%) "
                        "or lastModifiedDateTime ge %3%") %
            modifiedStr % nextByReceivedStr % nextByModifiedStr;
    return os.str();
}

} // namespace yrpopper::collector
