#pragma once

#include <collector_ng/http/datetime.h>
#include <common/context.h>
#include <ymod_httpclient/call.h>
#include <yplatform/json.h>

namespace yrpopper::collector {

using yplatform::json_value;

const char MS365_FOLDER_DELIM = '/';

struct MS365Folder
{
    std::string id;
    std::string name;
    std::uint64_t childFolderCount = 0;
    std::uint64_t totalItemCount = 0;
    std::string path;

    static MS365Folder fromJson(const json_value& src)
    {
        return MS365Folder{ src["id"].to_string(),
                            src["displayName"].to_string(),
                            src["childFolderCount"].to_uint64(),
                            src["totalItemCount"].to_uint64(),
                            "" };
    }
};
using MS365FolderList = std::vector<MS365Folder>;

struct MS365Message
{
    std::string id;
    std::string fid;
    std::string internetMessageId;
    std::time_t lastModifiedDateTime;
    std::time_t receivedDateTime;
    bool isRead = false;
    std::string subject;

    static MS365Message fromJson(const json_value& src)
    {
        return MS365Message{ src["id"].to_string(),
                             src["parentFolderId"].to_string(),
                             src["internetMessageId"].to_string(),
                             string_to_datetime(src["lastModifiedDateTime"].to_string()),
                             string_to_datetime(src["receivedDateTime"].to_string()),
                             src["isRead"].to_bool(),
                             src["subject"].to_string() };
    }

    friend bool operator<(const MS365Message& l, const MS365Message& r)
    {
        return std::tie(l.lastModifiedDateTime, l.receivedDateTime) <
            std::tie(r.lastModifiedDateTime, r.receivedDateTime);
    }
};
using MS365MessageList = std::vector<MS365Message>;

class MS365Client
{
public:
    MS365Client(
        rpop_context_ptr context,
        const std::string& accessToken,
        const std::string& baseUrl,
        std::uint32_t maxFetchSize);

    MS365FolderList fetchFolders();
    MS365MessageList fetchMessages(
        const std::string& fid,
        std::time_t prevModified,
        std::time_t prevReceived,
        std::uint64_t estimatedCount);
    std::string downloadMessage(const std::string& mid);

    virtual ~MS365Client() = default;

protected:
    virtual yhttp::response request(const std::string& url, std::string headers = "");

private:
    static const auto REQUEST_ALL = std::numeric_limits<std::size_t>::max();

    MS365Folder rootFolder();
    void fetchChildFolders(const MS365Folder& parent, MS365FolderList& output);

    std::string makeRootFolderUrl() const;
    std::string makeFetchChildFoldersUrl(const std::string& fid) const;
    std::string makeFetchMessagesUrl(
        const std::string& fid,
        std::time_t prevModified,
        std::time_t prevReceived) const;
    std::string makeDownloadMessageUrl(const std::string& mid) const;
    std::string makeMessageFilter(std::time_t modified, std::time_t received) const;

    rpop_context_ptr collectorCtx;
    std::string accessToken;
    std::string baseUrl;
    std::uint32_t maxFetchSize;

    template <typename Item>
    bool shouldAppendExtra(const std::vector<Item>& /*items*/, const Item& /*item*/)
    {
        return false;
    }

    template <>
    bool shouldAppendExtra(const std::vector<MS365Message>& items, const MS365Message& item)
    {
        if (items.empty()) return false;
        return items.back().lastModifiedDateTime == item.lastModifiedDateTime;
    }

    template <typename Item>
    std::vector<Item> requestMultiplePages(const std::string& url)
    {
        return requestMultiplePages<Item>(url, REQUEST_ALL);
    }

    template <typename Item>
    std::vector<Item> requestMultiplePages(const std::string& url, std::size_t estimatedCount)
    {
        std::vector<Item> items;
        if (estimatedCount < REQUEST_ALL) items.reserve(estimatedCount);

        bool shouldRequestNextPage = true;
        std::string nextLink = url;
        while (shouldRequestNextPage && !nextLink.empty())
        {
            auto response = request(nextLink);
            if (response.status != 200)
            {
                throw std::runtime_error(
                    "failed to make request for maltiple pages: status=" +
                    std::to_string(response.status) + " reason=" + response.reason);
            }

            json_value value;
            if (auto error = value.parse(response.body))
            {
                throw std::runtime_error("failed to parse http response body");
            }

            auto values = value["value"];
            auto parsed = parseItems(values, items, estimatedCount);
            if (parsed < values.size())
            {
                shouldRequestNextPage = false;
                // shouldRequestNextPage should be false if all required and extra items was
                // parsed. But if the next item is on the next page to meet stop condition we have
                // to make one more redundant request. To minimize count of such requests set
                // parameters: max_messages_per_folder % fetch_chunk_size != 0
            }

            nextLink = value["@odata.nextLink"].to_string("");
        }
        return items;
    }

    template <typename Item>
    std::size_t parseItems(
        const json_value& values,
        std::vector<Item>& outputItems,
        std::size_t estimatedCount)
    {
        std::size_t parsed = 0;
        for (auto&& json_item : values.array_items())
        {
            auto item = Item::fromJson(json_item);
            if (outputItems.size() < estimatedCount || shouldAppendExtra<Item>(outputItems, item))
            {
                outputItems.push_back(std::move(item));
                ++parsed;
            }
            else
            {
                break;
            }
        }
        return parsed;
    }
};
typedef std::shared_ptr<MS365Client> MS365ClientPtr;

}
