#pragma once

#include <collector_ng/http/ms365_client.h>
#include <collector_ng/collector_error.h>
#include <collector_ng/collector_settings.h>
#include <common/async/operation.h>
#include <common/typed_log.h>
#include <db/interface_provider.h>
#include <oauth/oauth.h>
#include <processor_ng/processor_service.h>

#include <ymod_httpclient/errors.h>
#include <yplatform/json.h>

namespace yrpopper::collector::operations {

typedef std::shared_ptr<boost::asio::io_service::strand> StrandPtr;

struct MS365CollectOpTraits
{
    using HttpClient = MS365Client;
    using DBInterface = db::HttpCollectorInterface;
    using DBInterfaceProvider = db::InterfaceProvider;
    using MessageProcessor = processor::ProcessorService;
};

inline std::string httpFolderToString(http_folder_ptr folder)
{
    std::stringstream ss;
    ss << "folder_name=" << folder->name
       << " id=" << folder->external_folder_id
       << " last_synced_message_ts=" << folder->last_synced_message_ts
       << " last_synced_message_received_ts=" << folder->last_synced_message_received_ts
       << " collected=" << folder->collected_count << " errors=" << folder->error_count
       << " bad_message_retries=" << folder->bad_message_retries;
    return ss.str();
}

template <typename Traits = MS365CollectOpTraits>
class MS365CollectOp
{
    typedef typename Traits::HttpClient HttpClient;
    typedef typename Traits::DBInterface DBInterface;
    typedef typename Traits::DBInterfaceProvider DBInterfaceProvider;
    typedef typename Traits::MessageProcessor MessageProcessor;
public:
    MS365CollectOp(rpop_context_ptr context, const CollectorSettings& settings)
        : collectorCtx(context), settings(settings)
    {
    }

    void execute()
    {
        initMS365Client();
        initDBInterface();
        doCollect();
    }

protected:
    void initMS365Client()
    {
        auto accessToken = getAccessToken();
        ms365Client = std::make_shared<HttpClient>(
            collectorCtx,
            accessToken,
            settings.httpCollector.ms365BaseUrl,
            settings.httpCollector.fetchChunkSize);
    }

    void initDBInterface()
    {
        dbInterface = DBInterfaceProvider::getHttpCollectorInterface();
    }

    void doCollect()
    {
        collectorCtx->sent_count = 0;

        auto folders = getFolders();
        for (auto&& [_, folder] : *folders)
        {
            processFolder(folder);
        }
    }

    std::string getAccessToken()
    {
        try
        {
            auto oauthModule = yplatform::find<oauth::OauthService>("oauth_module");
            auto futureRes = oauthModule->getAccessToken(
                collectorCtx, collectorCtx->task->server, collectorCtx->task->oauth_refresh_token);
            return futureRes.get();
        }
        catch (const std::exception& e)
        {
            throw AuthError(e.what());
        }
    }

    http_folders_ptr getFolders()
    {
        auto ms365Folders = fetchMS365Folders();
        auto dbFolders = loadFoldersFromDB();
        auto folders = boost::make_shared<http_folders>();
        for (auto&& outlookFolder : ms365Folders)
        {
            http_folder_ptr folder;
            auto it = dbFolders->find(outlookFolder.path);
            if (it != dbFolders->end())
            {
                folder = it->second;
            }
            else
            {
                // TODO: remove folder from DB
                folder = boost::make_shared<http_folder>();
            }
            folder->external_folder_id = outlookFolder.id;
            folder->name = outlookFolder.path;
            folder->message_count = outlookFolder.totalItemCount;
            folders->emplace(outlookFolder.path, folder);
        }
        return folders;
    }

    MS365FolderList fetchMS365Folders()
    {
        try
        {
            return ms365Client->fetchFolders();
        }
        catch (const std::exception& e)
        {
            handleMS365ClientError(std::current_exception());
            return {};
        }
    }

    http_folders_ptr loadFoldersFromDB()
    {
        try
        {
            return dbInterface->loadHttpFolders(collectorCtx, collectorCtx->task->popid).get();
        }
        catch (const std::exception& e)
        {
            handleDBError(std::current_exception());
            return {};
        }
    }

    void processFolder(http_folder_ptr folder)
    {
        TASK_LOG(collectorCtx, info)
            << "start process ms365 folder: " << httpFolderToString(folder);
        try
        {
            auto messages = fetchMessages(
                folder->external_folder_id,
                folder->last_synced_message_ts,
                folder->last_synced_message_received_ts);

            processMessages(messages, folder);
        }
        catch (const std::exception& e)
        {
            TASK_LOG(collectorCtx, error)
                << "failed process ms365 folder: " << httpFolderToString(folder)
                << " reason=" << e.what();
            updateOrCreateDbFolder(folder);
            throw;
        }
        updateOrCreateDbFolder(folder);
        TASK_LOG(collectorCtx, info)
            << "finish process ms365 folder: " << httpFolderToString(folder);
    }

    MS365MessageList fetchMessages(
        const std::string& fid,
        std::time_t sinceModified,
        std::time_t sinceReceived)
    {
        try
        {
            return ms365Client->fetchMessages(
                fid, sinceModified, sinceReceived, settings.httpCollector.maxMessagesPerFolder);
        }
        catch (const std::exception& e)
        {
            handleMS365ClientError(std::current_exception());
            return {};
        }
    }

    void processMessages(const MS365MessageList& messages, http_folder_ptr folder)
    {
        for (auto&& msg : messages)
        {
            try
            {
                processMessage(msg, folder->name);

                folder->last_synced_message_ts = msg.lastModifiedDateTime;
                folder->last_synced_message_received_ts = msg.receivedDateTime;
                folder->bad_message_retries = 0;
                ++folder->collected_count;
                ++collectorCtx->sent_count;
            }
            catch (const std::exception& e)
            {
                TASK_LOG(collectorCtx, error)
                    << "failed process ms365 message: id=" << msg.id << " reason=" << e.what();
                if (folder->bad_message_retries < settings.httpCollector.maxBadRetriesPerMessage)
                {
                    ++folder->bad_message_retries;
                    break;
                }
                else
                {
                    // TODO: add id to skipped_mids
                    folder->last_synced_message_ts = msg.lastModifiedDateTime;
                    folder->last_synced_message_received_ts = msg.receivedDateTime;
                    folder->bad_message_retries = 0;
                    ++folder->error_count;
                    TASK_LOG(collectorCtx, error)
                        << "skip ms365 message: id=" << msg.id << " reason=" << e.what();
                }
            }
        }
    }

    uint64_t updateOrCreateDbFolder(http_folder_ptr folder)
    {
        try
        {
            return dbInterface
                ->updateOrCreateHttpFolder(collectorCtx, collectorCtx->task->popid, folder)
                .get();
        }
        catch (const std::exception& e)
        {
            handleDBError(std::current_exception());
            return {};
        }
    }

    void processMessage(const MS365Message& srcMessage, const std::string& folder_name)
    {
        try
        {
            auto content = ms365Client->downloadMessage(srcMessage.id);
            auto message = prepareMessage(srcMessage, folder_name, std::move(content));

            auto processor = yplatform::find<MessageProcessor>("message_processor");
            auto res = processor->processMessage(collectorCtx, message);
            auto processResult = res.get();

            YLOG_CTX_GLOBAL(collectorCtx, info)
                << "processed message id=" << processResult.messageId
                << " smtpResponse=" << processResult.smtpResponse;
            reportProcessMessageSuccess(srcMessage, message, processResult);
        }
        catch (const std::exception& e)
        {
            reportProcessMessageFailure(srcMessage, e.what());
            throw;
        }
    }

    processor::MessagePtr prepareMessage(
        const MS365Message& msg,
        const std::string& folder_name,
        std::string&& content)
    {
        auto message = std::make_shared<processor::Message>();
        message->body = std::make_shared<std::string>(std::move(content));
        message->start = message->body->begin();
        message->end = message->body->end();
        message->seen = msg.isRead;
        message->srcFolder = folder_name;
        message->dstFolder = mapFolderName(folder_name, MS365_FOLDER_DELIM);
        message->spamFolder = message->dstFolder;
        message->dstDelim = MS365_FOLDER_DELIM;
        auto& label_id = collectorCtx->task->label_id;
        if (label_id.size())
        {
            message->lids.push_back(label_id);
        }
        return message;
    }

    std::string mapFolderName(const std::string& name, char delim)
    {
        auto pos = name.find(delim);
        auto topFolder = name.substr(0, pos);

        auto it = settings.folderNamesMap.find(topFolder);

        if (it != settings.folderNamesMap.end())
        {
            auto res = it->second;
            if (pos != string::npos)
            {
                res += name.substr(pos, string::npos);
            }
            return res;
        }
        return name;
    }

    void reportProcessMessageSuccess(
        const MS365Message& srcMessage,
        processor::MessagePtr processedMessage,
        const processor::ProcessorResult& processResult)
    {
        auto srcFolderName = processedMessage ? processedMessage->srcFolder : ""s;
        auto dstFolderName = processedMessage ? processedMessage->dstFolder : ""s;
        typed_log::log_store_message(
            collectorCtx,
            "success"s,
            ""s,
            dstFolderName,
            srcFolderName,
            srcMessage.id,
            "http"s,
            processResult.messageId,
            processResult.smtpResponse);
    }

    void reportProcessMessageFailure(const MS365Message& srcMessage, const std::string& error)
    {
        typed_log::log_store_message(
            collectorCtx,
            "error"s,
            error,
            ""s,
            ""s,
            srcMessage.id,
            "http"s,
            srcMessage.internetMessageId,
            ""s);
    }

    void handleMS365ClientError(std::exception_ptr e)
    {
        if (!e) return;
        try
        {
            std::rethrow_exception(e);
        }
        catch (const ymod_httpclient::resolve_error& e)
        {
            throw ResolveError(e.what());
        }
        catch (const ymod_httpclient::connect_error& e)
        {
            throw ConnectError(e.what());
        }
        catch (const ymod_httpclient::connection_timeout& e)
        {
            throw ConnectError(e.what());
        }
        catch (const ymod_httpclient::ssl_error& e)
        {
            throw SslError(e.what());
        }
        catch (const ymod_httpclient::request_timeout_error& e)
        {
            throw TimeoutError(e.what());
        }
        catch (const std::exception& e)
        {
            throw;
        }
    }

    void handleDBError(std::exception_ptr e)
    {
        if (!e) return;
        try
        {
            std::rethrow_exception(e);
        }
        catch (const std::exception& e)
        {
            throw DbError(e.what());
        }
    }

    rpop_context_ptr collectorCtx;
    const CollectorSettings& settings;
    std::shared_ptr<HttpClient> ms365Client;
    std::shared_ptr<DBInterface> dbInterface;
};

}
