#pragma once

#include <memory>

#include <common/context.h>
#include <common/uidl.h>
#include <common/async/operation.h>
#include <collector_ng/collector_settings.h>

#include <db/interface_provider.h>
#include <ymod_imapclient/imap_result.h>
#include <yplatform/future/future.hpp>
#include <db/imap_collector_interface.h>

namespace yrpopper { namespace collector {

class ImapFolder
{
public:
    ImapFolder(
        rpop_context_ptr context,
        imap_folder_ptr folder,
        ymod_imap_client::ImapListItemPtr mailbox,
        int32_t limit)
        : dbInterface(db::InterfaceProvider::getImapCollectorInterface())
        , context(context)
        , dbFolder(folder)
        , name(mailbox->name)
        , flags(mailbox->flags)
        , messageLimit(limit)

    {
    }

    enum MessageState
    {
        initialState,
        notCollectedState,
        errorState,
        collectedState,
        rejectedState
    };

    future_imap_uidl_map loadMessages(MessageState state, uint32_t count);
    future_void_t editMessage(uint32_t uid, MessageState newState);

    future_void_t addInitialMessages(std::vector<int64_t> messages);
    future_void_t save();

    bool isInited() const
    {
        return dbFolder->inited;
    }
    void setInited(bool inited)
    {
        dbFolder->inited = inited;
    }

    uint32_t getUidnext() const
    {
        return dbFolder->nextuid;
    }
    void setUidnext(uint32_t val)
    {
        dbFolder->nextuid = val;
    }

    uint32_t getMessageCount() const
    {
        return dbFolder->exists;
    }
    void setMessageCount(uint32_t val)
    {
        dbFolder->exists = val;
    }

    ymod_imap_client::Utf8MailboxName getName() const
    {
        return name;
    }
    uint32_t getFlags() const
    {
        return flags;
    }

    bool isValid() const
    {
        return valid;
    }
    void invalidate()
    {
        this->valid = false;
    }

    size_t getMessageLimit()
    {
        return messageLimit;
    }
    void updateLimit(size_t collected)
    {
        if (messageLimit <= collected)
        {
            messageLimit = 0;
        }
        else
        {
            messageLimit -= collected;
        }
    }

private:
    db::ImapCollectorInterfacePtr dbInterface;
    rpop_context_ptr context;
    imap_folder_ptr dbFolder;

    ymod_imap_client::Utf8MailboxName name;
    uint32_t flags;

    size_t messageLimit;

    bool valid = true;

    static const std::map<MessageState, std::string> stringStates;
};

typedef std::shared_ptr<ImapFolder> ImapFolderPtr;
typedef yplatform::future::future<ImapFolderPtr> FutureImapFolder;
typedef yplatform::future::promise<ImapFolderPtr> PromiseImapFolder;

class ImapFolderList;
typedef std::shared_ptr<ImapFolderList> ImapFolderListPtr;

class ImapFolderList
{
public:
    static ImapFolderListPtr get(rpop_context_ptr context, CollectorSettingsPtr settings);
    virtual ~ImapFolderList()
    {
    }

    virtual future_void_t init() = 0;
    virtual FutureImapFolder prepareFolder(
        ymod_imap_client::ImapListItemPtr mailbox,
        ymod_imap_client::ImapMailboxResultPtr mailboxStats) = 0;

    virtual void markProcessed() = 0;

    const std::vector<ImapFolderPtr>& getPrepared() const
    {
        return imapFolders;
    }

protected:
    ImapFolderList(rpop_context_ptr context, CollectorSettingsPtr settings)
        : dbInterface(db::InterfaceProvider::getImapCollectorInterface())
        , context(context)
        , settings(settings)
    {
    }

    db::ImapCollectorInterfacePtr dbInterface;
    rpop_context_ptr context;
    CollectorSettingsPtr settings;

    imap_folders_ptr dbFolders;

    std::vector<ImapFolderPtr> imapFolders;
};

} // namespace collector
} // namespace yrpopper
