#include <collector_ng/imap/imap_folders.h>
#include <collector_ng/imap/imap_folders_impl.h>

namespace yrpopper::collector {

const std::map<ImapFolder::MessageState, std::string> ImapFolder::stringStates{
    { initialState, "initial" },
    { notCollectedState, "not_collected" },
    { errorState, "error" },
    { collectedState, "collected" },
    { rejectedState, "rejected" }
};

ImapFolderListPtr ImapFolderList::get(rpop_context_ptr context, CollectorSettingsPtr settings)
{
    return std::make_shared<ImapFolderListImpl>(context, settings);
}

void ImapFolderListImpl::processFutureFolders(promise_void_t res, future_imap_folders folders)
{
    try
    {
        setDbFolders(folders.get());
        res.set(VoidResult());
        inited = true;
    }
    catch (...)
    {
        res.set_current_exception();
        inited = false;
    }
}
future_void_t ImapFolderListImpl::init()
{
    promise_void_t res;

    imapFolders.clear();
    inited = false;

    auto folders = loadFoldersFromDB();
    auto self = shared_from_this();
    folders.add_callback([=]() { self->processFutureFolders(res, folders); });

    return res;
}

future_imap_folders ImapFolderListImpl::loadFoldersFromDB()
{
    return dbInterface->loadImapFolders(context, context->task->popid);
}

future_uint64_t ImapFolderListImpl::createFolderDB(imap_folder_ptr folder)
{
    YRIMAP_LOG(context) << "folder " << folder->name
                        << " is new, uidvalidity=" << folder->uidvalidity;
    return dbInterface->appendImapFolder(context, context->task->popid, folder);
}

future_void_t ImapFolderListImpl::deleteFolderDB(imap_folder_ptr folder)
{
    YRIMAP_LOG(context) << "folder " << folder->name
                        << " deleted(uidvalidity changed), old=" << folder->uidvalidity;
    imap_folders_ptr toDeletion = boost::make_shared<imap_folders>();
    (*toDeletion)[folder->name] = folder;

    return dbInterface->deleteImapFolders(context, context->task->popid, toDeletion);
}

ImapFolderPtr ImapFolderListImpl::createImapFolder(imap_folder_ptr folder, ImapListItemPtr mailbox)
{
    auto imapFolder =
        std::make_shared<ImapFolder>(context, folder, mailbox, settings->maxMessagesPerFolder);
    imapFolders.push_back(imapFolder);

    return imapFolder;
}

void ImapFolderListImpl::setDbFolders(imap_folders_ptr folders)
{
    dbFolders = folders;
}

FutureImapFolder ImapFolderListImpl::prepareFolder(
    ImapListItemPtr mailbox,
    ImapMailboxResultPtr mailboxStats)
{
    auto prepareHelper = std::make_shared<PrepareHelper>();
    prepareHelper->folderList = shared_from_this();
    prepareHelper->mailbox = mailbox;
    prepareHelper->mailboxStats = mailboxStats;

    auto self = shared_from_this();
    auto callback = [=](std::exception_ptr e) { self->handlePreapareResult(e, prepareHelper); };
    runAsync<PrepareImapFolder>(callback, prepareHelper);

    return prepareHelper->promise;
}

void ImapFolderListImpl::handlePreapareResult(std::exception_ptr e, PrepareHelperPtr helper)
{
    if (e)
    {
        helper->promise.set_exception(e);
    }
    else
    {
        helper->promise.set(helper->folder);
    }
}

future_imap_uidl_map ImapFolder::loadMessages(MessageState state, uint32_t count)
{
    return dbInterface->loadImapMessages(
        context, dbFolder->folder_id, stringStates.at(state), count);
}

future_void_t ImapFolder::editMessage(uint32_t uid, MessageState newState)
{
    return dbInterface->editImapMessage(
        context, dbFolder->folder_id, uid, stringStates.at(newState));
}

future_void_t ImapFolder::addInitialMessages(std::vector<int64_t> messages)
{
    return dbInterface->addInitialImapMessages(context, dbFolder->folder_id, std::move(messages));
}

future_void_t ImapFolder::save()
{
    return dbInterface->editImapFolder(context, dbFolder);
}

} // namespace yrpopper::collector
