#pragma once

#include <collector_ng/imap/operations/imap_operation.h>
#include <collector_ng/imap/imap_folders.h>

using ymod_imap_client::ImapListItemPtr;
using ymod_imap_client::ImapMailboxResultPtr;

namespace yrpopper { namespace collector {

struct PrepareHelper;
typedef std::shared_ptr<PrepareHelper> PrepareHelperPtr;

class ImapFolderListImpl
    : public std::enable_shared_from_this<ImapFolderListImpl>
    , public ImapFolderList
{
public:
    ImapFolderListImpl(rpop_context_ptr context, CollectorSettingsPtr settings)
        : ImapFolderList(context, settings)
    {
    }

    bool isInited() const
    {
        return inited;
    }
    void setInited(bool inited)
    {
        this->inited = inited;
    }

    virtual future_void_t init() override;
    virtual FutureImapFolder prepareFolder(
        ymod_imap_client::ImapListItemPtr mailbox,
        ymod_imap_client::ImapMailboxResultPtr mailboxStats) override;

    virtual void markProcessed() override
    {
        for (auto& folder : imapFolders)
        {
            if (!folder->isValid())
            {
                setInited(false);
                break;
            }
        }

        imapFolders.clear();
    }

    future_uint64_t createFolderDB(imap_folder_ptr folder);
    future_void_t deleteFolderDB(imap_folder_ptr folder);

    ImapFolderPtr createImapFolder(imap_folder_ptr folder, ImapListItemPtr mailbox);

    future_imap_folders loadFoldersFromDB();
    void setDbFolders(imap_folders_ptr folders);
    imap_folders_ptr getDbFolders() const
    {
        return dbFolders;
    }

private:
    void processFutureFolders(promise_void_t res, future_imap_folders folders);
    void handlePreapareResult(std::exception_ptr e, PrepareHelperPtr helper);

    bool inited = false;
};
typedef std::shared_ptr<ImapFolderListImpl> ImapFolderListImplPtr;

struct PrepareHelper
{
    PromiseImapFolder promise;

    ImapListItemPtr mailbox;
    ImapMailboxResultPtr mailboxStats;
    ImapFolderListImplPtr folderList;

    ImapFolderPtr folder;
};

template <class Operation>
using PrepareOpBuilder = operations::ImapOpBuilder<Operation, PrepareHelperPtr>;

class InitFolderList : public FutureOperation<VoidResult>
{
public:
    InitFolderList(AsyncCallback cb, PrepareHelperPtr helper)
        : FutureOperation<VoidResult>(cb), helper(helper)
    {
    }

protected:
    virtual FutureType getFuture() override
    {
        return helper->folderList->init();
    }

private:
    PrepareHelperPtr helper;
};

class InitFolderListConditional : public ConditionalOperation<InitFolderList>
{
public:
    InitFolderListConditional(AsyncCallback cb, PrepareHelperPtr helper)
        : ConditionalOperation<InitFolderList>(cb, helper), helper(helper)
    {
    }

protected:
    virtual bool checkCondition() override
    {
        return !helper->folderList->isInited();
    }

private:
    PrepareHelperPtr helper;
};

class DeleteDbFolder : public FutureOperation<VoidResult>
{
public:
    DeleteDbFolder(AsyncCallback cb, PrepareHelperPtr helper)
        : FutureOperation<VoidResult>(cb), helper(helper)
    {
    }

protected:
    virtual FutureType getFuture() override
    {
        auto it = helper->folderList->getDbFolders()->find(helper->mailbox->name.asString());
        dbFolder = it->second;

        helper->folderList->getDbFolders()->erase(it);
        return helper->folderList->deleteFolderDB(dbFolder);
    }

    virtual std::exception_ptr onFinish(std::exception_ptr e) override
    {
        if (e)
        {
            // error deleting from DB, return to map
            (*helper->folderList->getDbFolders())[dbFolder->name] = dbFolder;
        }
        return e;
    }

private:
    PrepareHelperPtr helper;
    imap_folder_ptr dbFolder;
};

class DeleteDbFolderConditional : public ConditionalOperation<DeleteDbFolder>
{
public:
    DeleteDbFolderConditional(AsyncCallback cb, PrepareHelperPtr helper)
        : ConditionalOperation<DeleteDbFolder>(cb, helper), helper(helper)
    {
    }

protected:
    virtual bool checkCondition() override
    {
        auto it = helper->folderList->getDbFolders()->find(helper->mailbox->name.asString());
        if (it != helper->folderList->getDbFolders()->end())
        {
            return it->second->uidvalidity != helper->mailboxStats->mailboxInfo.uidvalidity_;
        }

        return false;
    }

private:
    PrepareHelperPtr helper;
};

class CreateDbFolder : public FutureOperation<uint64_t>
{
public:
    CreateDbFolder(AsyncCallback cb, PrepareHelperPtr helper)
        : FutureOperation<uint64_t>(cb), helper(helper)
    {
    }

protected:
    virtual FutureType getFuture() override
    {
        folder = boost::make_shared<imap_folder>();
        auto& mailboxStats = helper->mailboxStats;

        folder->folder_id = 0;
        folder->name = helper->mailbox->name.asString();
        folder->uidvalidity = mailboxStats->mailboxInfo.uidvalidity_;
        folder->nextuid = 1;
        folder->exists = mailboxStats->mailboxInfo.exists_;
        folder->inited = true;

        return helper->folderList->createFolderDB(folder);
    }

    virtual void process(uint64_t&& res) override
    {
        folder->folder_id = res;
        (*helper->folderList->getDbFolders())[folder->name] = folder;
    }

private:
    PrepareHelperPtr helper;
    imap_folder_ptr folder;
};

class CreateDbFolderConditional : public ConditionalOperation<CreateDbFolder>
{
public:
    CreateDbFolderConditional(AsyncCallback cb, PrepareHelperPtr helper)
        : ConditionalOperation<CreateDbFolder>(cb, helper), helper(helper)
    {
    }

protected:
    virtual bool checkCondition() override
    {
        auto it = helper->folderList->getDbFolders()->find(helper->mailbox->name.asString());
        return (it == helper->folderList->getDbFolders()->end());
    }

private:
    PrepareHelperPtr helper;
};

class PrepareFolder : public ProxyOperation<CreateDbFolderConditional>
{
public:
    PrepareFolder(AsyncCallback cb, PrepareHelperPtr helper)
        : ProxyOperation<CreateDbFolderConditional>(cb), helper(helper)
    {
    }

protected:
    void start() override
    {
        initProxy(helper);
        run();
    }

    virtual std::exception_ptr onFinish(std::exception_ptr e) override
    {
        if (!e)
        {
            auto it = helper->folderList->getDbFolders()->find(helper->mailbox->name.asString());
            helper->folder = helper->folderList->createImapFolder(it->second, helper->mailbox);
        }
        return e;
    }

private:
    PrepareHelperPtr helper;
};

class UpdateDbFolder : public FutureOperation<VoidResult>
{
public:
    UpdateDbFolder(AsyncCallback cb, PrepareHelperPtr helper)
        : FutureOperation<VoidResult>(cb), helper(helper)
    {
    }

protected:
    virtual FutureType getFuture() override
    {
        folder = helper->folder;

        folder->setUidnext(1);
        folder->setInited(true);

        return folder->save();
    }

    virtual std::exception_ptr onFinish(std::exception_ptr e) override
    {
        if (e)
        {
            folder->invalidate();
        }
        return e;
    }

private:
    PrepareHelperPtr helper;
    ImapFolderPtr folder;
};

class UpdateDbFolderConditional : public ConditionalOperation<UpdateDbFolder>
{
public:
    UpdateDbFolderConditional(AsyncCallback cb, PrepareHelperPtr helper)
        : ConditionalOperation<UpdateDbFolder>(cb, helper), helper(helper)
    {
    }

protected:
    virtual bool checkCondition() override
    {
        auto folder = helper->folder;
        return (!folder->isInited());
    }

private:
    PrepareHelperPtr helper;
};

typedef ComplexOperation<
    PrepareHelperPtr,
    PrepareOpBuilder,

    InitFolderListConditional,
    DeleteDbFolderConditional,
    PrepareFolder,
    UpdateDbFolderConditional>
    PrepareImapFolder;

} // namespace collector
} // namespace yrpopper
