#pragma once

#include <collector_ng/imap/operations/imap_operation.h>
#include <collector_ng/imap/operations/imap_folder_operations.h>
#include <collector_ng/imap/operations/imap_message_operations.h>

#include <collector_ng/imap/helpers/imap_collect_folders_helper.h>

#include <ymod_imapclient/call.h>

namespace yrpopper { namespace collector { namespace operations {

template <typename T>
using ImapCollectFoldersOpBuilder = ImapOpBuilder<T, ImapCollectFoldersHelperPtr>;

template <typename Res>
using ImapCollectFoldersOperation = GeneralImapOperation<ImapCollectFoldersHelperPtr, Res>;

class ImapExamine : public ImapCollectFoldersOperation<ymod_imap_client::ImapMailboxResultPtr>
{
public:
    ImapExamine(AsyncCallback cb, ImapCollectFoldersHelperPtr helper)
        : ImapCollectFoldersOperation<ymod_imap_client::ImapMailboxResultPtr>(cb, helper)
    {
    }

protected:
    virtual FutureType getFuture() override
    {
        auto imapClient = helper->getImapClient();
        return imapClient->examine(helper->getCurrentMailbox()->name);
    }

    virtual void process(ymod_imap_client::ImapMailboxResultPtr&& res) override
    {
        helper->setCurrentMailboxStats(res);
    }
};

class ImapGetFolderStatus : public ImapCollectFoldersOperation<ymod_imap_client::ImapStatusPtr>
{
public:
    ImapGetFolderStatus(AsyncCallback cb, ImapCollectFoldersHelperPtr helper)
        : ImapCollectFoldersOperation<ymod_imap_client::ImapStatusPtr>(cb, helper)
    {
    }

protected:
    virtual FutureType getFuture() override
    {
        auto imapClient = helper->getImapClient();
        TASK_LOG(helper->getContext(), info) << "Getting uidnext from status";
        return imapClient->status(helper->getCurrentMailbox()->name, "UIDNEXT");
    }

    virtual void process(ymod_imap_client::ImapStatusPtr&& res) override
    {
        helper->getCurrentMailboxStats()->mailboxInfo.nextuid_ = res->mailboxInfo.nextuid_;
    }
};

class ImapConditionalFolderStatus : public ConditionalOperation<ImapGetFolderStatus>
{
public:
    ImapConditionalFolderStatus(AsyncCallback cb, ImapCollectFoldersHelperPtr helper)
        : ConditionalOperation<ImapGetFolderStatus>(cb, helper), helper(helper)
    {
    }

protected:
    virtual bool checkCondition() override
    {
        return helper->getCurrentMailboxStats()->mailboxInfo.nextuid_ == 0;
    }

private:
    ImapCollectFoldersHelperPtr helper;
};

class ImapPrepareFolder : public ImapCollectFoldersOperation<ImapFolderPtr>
{
public:
    ImapPrepareFolder(AsyncCallback cb, ImapCollectFoldersHelperPtr helper)
        : ImapCollectFoldersOperation<ImapFolderPtr>(cb, helper)
    {
    }

protected:
    virtual FutureType getFuture() override
    {
        auto folderList = helper->getImapFolderList();
        return folderList->prepareFolder(
            helper->getCurrentMailbox(), helper->getCurrentMailboxStats());
    }

    virtual void process(ImapFolderPtr&& res) override
    {
        helper->setImapFolder(res);
    }
};

class ImapFetchMessages : public ProxyOperation<ImapFetchFolderMessagesAsyncOperation>
{
public:
    ImapFetchMessages(AsyncCallback cb, ImapCollectFoldersHelperPtr helper)
        : ProxyOperation<ImapFetchFolderMessagesAsyncOperation>(cb), helper(helper)
    {
    }

    void start()
    {
        auto folderHelper = std::make_shared<ImapFolderHelper>(
            *helper, helper->getImapFolder(), helper->getCurrentMailboxStats());

        initProxy(folderHelper);
        run();
    }

private:
    ImapCollectFoldersHelperPtr helper;
};

class ImapProcessFolder : public ProxyOperation<ImapProcessMessages>
{
public:
    ImapProcessFolder(AsyncCallback cb, ImapCollectFoldersHelperPtr helper)
        : ProxyOperation<ImapProcessMessages>(cb), listHelper(helper)
    {
    }

    void start()
    {
        auto messageHelper = std::make_shared<ImapMessageHelper>(
            *listHelper, listHelper->getImapFolder(), ImapFolder::initialState);

        initProxy(messageHelper);
        run();
    }

private:
    ImapCollectFoldersHelperPtr listHelper;
};

typedef ComplexOperation<
    ImapCollectFoldersHelperPtr,
    ImapCollectFoldersOpBuilder,

    ImapExamine,
    ImapConditionalFolderStatus,
    ImapPrepareFolder,
    ImapFetchMessages,
    ImapProcessFolder>
    ImapCollectFolder;

class ImapFolderIteration : public ProxyOperation<ImapCollectFolder>
{
public:
    ImapFolderIteration(AsyncCallback cb, ImapCollectFoldersHelperPtr helper)
        : ProxyOperation<ImapCollectFolder>(cb), listHelper(helper)
    {
    }

    void start() override
    {
        TASK_LOG(listHelper->getContext(), info)
            << "Processing folder \"" << listHelper->getCurrentMailbox()->name.asString() << "\"";

        initProxy(listHelper);
        run();
    }

protected:
    virtual std::exception_ptr onFinish(std::exception_ptr err) override
    {
        if (err)
        {
            try
            {
                std::rethrow_exception(err);
            }
            catch (const ymod_imap_client::NoException& e)
            {
                TASK_LOG(listHelper->getContext(), error)
                    << "ImapFolderIteration exception: " << e.what();
                err = std::exception_ptr();
            }
            catch (const ymod_imap_client::BadException& e)
            {
                TASK_LOG(listHelper->getContext(), error)
                    << "ImapFolderIteration exception: " << e.what();
                err = std::exception_ptr();
            }
            catch (const std::exception& e)
            {
                return err;
            }
        }
        listHelper->nextMailbox();
        return err;
    }

private:
    ImapCollectFoldersHelperPtr listHelper;
};

class ImapFoldersCycle : public CycleOperation<ImapFolderIteration>
{
public:
    ImapFoldersCycle(AsyncCallback cb, ImapCollectFoldersHelperPtr helper)
        : CycleOperation<ImapFolderIteration>(cb, helper), helper(helper)
    {
    }

protected:
    virtual bool checkCondition() override
    {
        return helper->hasMoreMailboxes();
    }

private:
    ImapCollectFoldersHelperPtr helper;
};

} // namespace operations
} // namespace collector
} // namespace yrpopper
